package boccuz.rnn;

import java.util.HashMap;
import java.util.Map;

import org.newdawn.slick.Animation;
import org.newdawn.slick.Graphics;
import org.newdawn.slick.Image;
import org.newdawn.slick.SlickException;
import org.newdawn.slick.SpriteSheet;

public class Entity implements Entity2I{
	/** The character's animations */
	private Map<Direction, Animation> animations;
	/** The sprite sheet reference */
	private String spriteRef;
	/** The current direction */
	private Direction currentDirection;
	/** The entity position */
	public Vector2I pos;
	/** The movement */
	public Vector2I moveVector;
	/** The tile width */
	private int tw;
	/** The tile height */
	private int th;

	public Entity(final String ref, int tw, int th) throws SlickException {
		this.spriteRef = ref;
		this.currentDirection = Direction.DOWN;
		this.tw = tw;
		this.th = th;
		this.pos = new Vector2I(0, 0);
		createAnimations(false, 200);
	}

	/**
	 * Create the character animations
	 * 
	 * @param autoAnim
	 *            - true to animate
	 * @param speed
	 *            - the animation update speed in ms
	 * @throws SlickException
	 */
	private void createAnimations(final boolean autoAnim, final int speed)
			throws SlickException {
		Image img = Asset.getImage(spriteRef);
		SpriteSheet ss = new SpriteSheet(img, tw, th);

		animations = new HashMap<Direction, Animation>();
		createAnimation(Direction.DOWN, ss, autoAnim, speed);
		createAnimation(Direction.LEFT, ss, autoAnim, speed);
		createAnimation(Direction.RIGHT, ss, autoAnim, speed);
		createAnimation(Direction.UP, ss, autoAnim, speed);
	}

	/**
	 * Create a single animation for a direction
	 * 
	 * @param d
	 *            - the direction
	 * @param ss
	 *            - the sprite sheet source
	 * @param autoAnim
	 *            - true to animate
	 * @param speed
	 *            - the animation update speed in ms
	 */
	private void createAnimation(final Direction d, final SpriteSheet ss,
			final boolean autoAnim, final int speed) {
		final int row = d.getSpriteY();
		Animation a = new Animation();
		a.setAutoUpdate(autoAnim);
		a.addFrame(ss.getSprite(1, row), speed);
		a.addFrame(ss.getSprite(0, row), speed);
		a.addFrame(ss.getSprite(1, row), speed);
		a.addFrame(ss.getSprite(0, row), speed);
		animations.put(d, a);
	}

	/**
	 * @return the character's animation according to it's current direction
	 */
	public Animation getAnimation() {
		return animations.get(currentDirection);
	}

	public Direction getCurrentDirection() {
		return currentDirection;
	}

	public void setMoving(boolean move) {
		for (Direction d : Direction.values()) {
			animations.get(d).setAutoUpdate(move);
		}
	}

	public void setCurrentDirection(Direction currentDirection) {
		this.currentDirection = currentDirection;
	}

	@Override
	public String toString() {
		return "Entity [animations=" + animations + ", spriteRef=" + spriteRef
				+ ", currentDirection=" + currentDirection + ", pos=" + pos
				+ "]";
	}

	@Override
	public Vector2I getPosition() {
		return pos;
	}

	@Override
	public void draw(Vector2I pos, Graphics g) {
		getAnimation().draw(pos.x, pos.y);
	}
}
