/**
 * Bachelorarbeit: Crowdsourcing- Systeme Manipulation und Resistenz
 * von Jens Bothur
 */
package jens.bothur.occt.automatons;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import jens.bothur.occt.domainobjects.RateableObject;
import jens.bothur.occt.domainobjects.User;
import jens.bothur.occt.domainvalues.AverageReview;
import jens.bothur.occt.domainvalues.Review;
import jens.bothur.occt.services.IRateableObjectService;

/**
 * Diese Klasse reprsentiert den Berechner fr durchschnittliche Bewertungen.
 * Diese Klasse hat nur eine nach auen sichtbare Methode. Stt man diese
 * Methode an werden ALLE durchschnittlichen Bewertungen ALLE bewertbaren
 * Objekte im System mit dem aktuellen Datenstand neu berechnet.
 * 
 * @author Jens Bothur
 */
public class AverageRatingCalculator {

	/**
	 * Eine {@link List}e mit allen bewertbaren Objekten (
	 * {@link RateableObject} ).
	 */
	private Collection<RateableObject> _rateableObjects;

	/**
	 * Der zu benutzene {@link IRateableObjectService}.
	 */
	private IRateableObjectService _objectService;

	/**
	 * Konstruktor fr {@link AverageRatingCalculator}.
	 * 
	 * @param objectService
	 *            Der zu benutzene {@link IRateableObjectService}.
	 */
	public AverageRatingCalculator(IRateableObjectService objectService) {
		_objectService = objectService;
	}

	/**
	 * Dieser Methode veranlasst eine Neuberechnung aller durchschnittlichen
	 * Bewertungen von alle bewertbaren Objekten im System.
	 */
	public void calculateAverageRatings() {
		_rateableObjects = _objectService.getAllRateableObjects();
		for (RateableObject object : _rateableObjects) {
			calculateAverageRating(object);
		}
	}

	/**
	 * Diese Methode berechnet die durchschnittliche Bewertung fr ein gegebenes
	 * bewertbares Objekt neu und setz es an diesem Objekt.
	 * 
	 * @param object
	 *            Das {@link RateableObject} fr das die durchschnittliche
	 *            Bewertung neu berechnet werden soll.
	 */
	private void calculateAverageRating(RateableObject object) {
		Set<User> raters = object.getReviewers();
		int sumOfWeights = 0;
		int sumOfRatings = 0;

		if (!raters.isEmpty()) {
			for (User user : raters) {
				int confidenceLevelValue = user.getConfidenceLevel()
						.getValueTimes10();
				Review review = object.getReviewForReviewer(user);
				sumOfWeights += confidenceLevelValue;
				sumOfRatings += confidenceLevelValue * review.getRating();
			}

			sumOfRatings = sumOfRatings * 10;
			int averageRatingTimes10 = 35;
			if (sumOfWeights != 0) {
				averageRatingTimes10 = sumOfRatings / sumOfWeights;
			}

			AverageReview averageReview = AverageReview
					.getAverageReviewByIntTimes10(averageRatingTimes10);
			object.setAverageReview(averageReview);
		}

	}

}
