/**
 * Bachelorarbeit: Crowdsourcing- Systeme Manipulation und Resistenz
 * von Jens Bothur
 */
package jens.bothur.occt.automatons.security_framework.group_calculators;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

import jens.bothur.occt.domainobjects.RateableObject;
import jens.bothur.occt.domainobjects.ReviewerGroup;

/**
 * Diese Klasse reprsentiert einen Berechner fr das Gruppen-Gren-Verhltnis.
 * Dabei geht es darum die Gre einer Bewerter-Gruppe mit der gesamten Anzahl
 * an Bewertungen fr bewertbarn Objekte, welche diese Gruppe bewertet hat, zu
 * vergleichen. Am Ende wird fr alle Objekte dieser Gruppe der Durchschnitt
 * gebildet.
 * 
 * @author Jens Bothur
 */
public class GroupSizeRatioSpamScoreCalculator implements
		IGroupSpamScoreCalculator {

	/**
	 * Berechnet das Gruppen-Gren-Verhltnis fr eine Menge von
	 * Bewerter-Gruppen.
	 * 
	 * @param reviewerGroups
	 *            Ein {@link Set} von {@link ReviewerGroup}.
	 * @return Eine {@link Map} welche allen reviewerGroups ihr berechnetes
	 *         Gruppen-Gren-Verhltnis als {@link Double} zuordnet.
	 */
	public static Map<ReviewerGroup, Double> calculateGroupSizeRatioSpamScore(
			Set<ReviewerGroup> reviewerGroups) {
		Map<ReviewerGroup, Double> result = new LinkedHashMap<ReviewerGroup, Double>();

		for (ReviewerGroup reviewerGroup : reviewerGroups) {
			double score = calculateGroupSizeRatioSpamScoreForGroup(reviewerGroup);
			result.put(reviewerGroup, score);

		}

		return result;

	}

	/**
	 * Berechnet das Gruppen-Gren-Verhltnis fr eine bergebene Gruppe.
	 * 
	 * @param reviewerGroup
	 *            Eine {@link ReviewerGroup} fr die berechnet werden soll.
	 * @return Das berechnete Gruppen-Gren-Verhltnis fr die reviewerGroup.
	 */
	private static double calculateGroupSizeRatioSpamScoreForGroup(
			ReviewerGroup reviewerGroup) {
		double result = 0.0;

		Set<RateableObject> collaboratedObjects = reviewerGroup
				.getCollaboratedObjects();
		for (RateableObject rateableObject : collaboratedObjects) {
			double score = calculateGroupSizeRatioSpamScoreForGroupAndObject(
					reviewerGroup, rateableObject);
			result += score;
		}

		result = result / (double) collaboratedObjects.size();

		return result;
	}

	/**
	 * Berechnet das Gruppen-Gren-Verhltnis fr eine bergebene Gruppe
	 * bezglich eines bergebenen bewertbaren Objektes.
	 * 
	 * @param reviewerGroup
	 *            Eine {@link ReviewerGroup} fr die berechnet werden soll.
	 * @param rateableObject
	 *            Ein {@link RateableObject} welches die {@link ReviewerGroup}
	 *            bewertet hat.
	 * @return Das berechnetete Gruppen-Gren-Verhltnis fr die reviewerGroup
	 *         bezglich des rateableObjects.
	 */
	private static double calculateGroupSizeRatioSpamScoreForGroupAndObject(
			ReviewerGroup reviewerGroup, RateableObject rateableObject) {
		double membersSize = reviewerGroup.getMembers().size();
		double totalReviewerSize = rateableObject.getReviewers().size();

		double result = membersSize / totalReviewerSize;
		return result;
	}

}
