/**
 * Bachelorarbeit: Crowdsourcing- Systeme Manipulation und Resistenz
 * von Jens Bothur
 */
package jens.bothur.occt.automatons.security_framework.group_calculators;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import jens.bothur.occt.domainobjects.RateableObject;
import jens.bothur.occt.domainobjects.ReviewerGroup;
import jens.bothur.occt.domainobjects.User;

/**
 * Diese Klasse reprsentiert einen Berechner fr den Gruppen-Spam-Score
 * bezglich des Zeitfensters. Hiermit ist gemeint, dass hchstwahrscheinlich
 * eine Bewerter-Gruppe ihre gemeinsam bewerteten Objekte wahrscheinlich
 * innerhalb von kurzer Zeit bewertet hat. Das Zeitfenster hat eine lnge von
 * {@value #TIME_WINDOW_LENGTH} Stunden.
 * 
 * @author Jens Bothur
 */
public class TimeWindowSpamScoreCalculator implements IGroupSpamScoreCalculator {

	/**
	 * Die Lnge des Zeitfensters in Stunden.
	 */
	private static final int TIME_WINDOW_LENGTH = 12;

	/**
	 * Berechnet den Gruppen-Zeitfenset-SpamScore fr eine bergebene Menge von
	 * Bewerter-Gruppen.
	 * 
	 * @param reviewerGroups
	 *            Ein {@link Set} von {@link ReviewerGroup}s.
	 * @return Eine {@link Map} welche jede {@link ReviewerGroup zu ihrem
	 *         berechneten Score als {@link Double} zuweist.
	 */
	public static Map<ReviewerGroup, Double> calculateTimeWindowSpamScore(
			Set<ReviewerGroup> reviewerGroups) {
		Map<ReviewerGroup, Double> result = new LinkedHashMap<ReviewerGroup, Double>();
		for (ReviewerGroup reviewerGroup : reviewerGroups) {
			result.put(reviewerGroup,
					calculateTimeWindowSpamScoreForGroup(reviewerGroup));
		}

		return result;
	}

	/**
	 * Berechnet den Gruppen-Zeitfenster-SpamScore fr eine bergebene
	 * Bewerter-Gruppe.
	 * 
	 * @param reviewerGroup
	 *            Eine {@link ReviewerGroup}.
	 * @return Der berechente Score der reviewerGroup als {@link Double}.
	 */
	private static Double calculateTimeWindowSpamScoreForGroup(
			ReviewerGroup reviewerGroup) {
		double result = 0.0;

		for (RateableObject rateableObject : reviewerGroup
				.getCollaboratedObjects()) {
			double objectScore = calculateTimeWindowSpamScoreForGroupAndObject(
					reviewerGroup, rateableObject);
			if (objectScore > result) {
				result = objectScore;
			}
		}

		return result;
	}

	/**
	 * Berechnet den Gruppen-Zeitfenster-SpamScore einer Bewerter-Gruppe
	 * bezglich eines durch die Gruppe bewerteten Objektes.
	 * 
	 * @param reviewerGroup
	 *            Eine {@link ReviewerGroup}.
	 * @param rateableObject
	 *            Ein durch reviewerGroup bewertetes {@link RateableObject}.
	 * @return Der berechnete Score als {@link Double}.
	 */
	private static double calculateTimeWindowSpamScoreForGroupAndObject(
			ReviewerGroup reviewerGroup, RateableObject rateableObject) {
		Set<User> members = reviewerGroup.getMembers();
		List<Date> ratingTimes = new ArrayList<Date>();
		for (User user : members) {
			ratingTimes.add(rateableObject.getDateForReviewer(user));
		}
		Collections.sort(ratingTimes);

		double result = 0.0;
		double timeWindow = 60 * 60 * 1000 * TIME_WINDOW_LENGTH;
		Date firstRating = ratingTimes.get(0);
		Date lastRating = ratingTimes.get(ratingTimes.size() - 1);
		double timeDiff = lastRating.getTime() - firstRating.getTime();
		if (timeDiff < timeWindow) {
			result = 1.0 - (timeDiff / timeWindow);
		}

		return result;
	}

}
