/**
 * Bachelorarbeit: Crowdsourcing- Systeme Manipulation und Resistenz
 * von Jens Bothur
 */
package jens.bothur.occt.automatons.security_framework.single_calculators;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import jens.bothur.occt.domainobjects.GroupOfRateableObjects;
import jens.bothur.occt.domainobjects.RateableObject;
import jens.bothur.occt.domainobjects.User;
import jens.bothur.occt.domainvalues.Review;

/**
 * Diese Klasse reprsentiert eine Bewertungs-Spam-Score Berechner. Hierbei wird
 * untersucht ob eine bestimmte Gruppe von hnlichen bewertbaren Objekten von
 * einem Benutzer besonders hufig gleich bewertet wurde. Fr eine Gruppe werden
 * die Spam-Scores dann in Verhltnis zueinander gesetzt (der hchste bekommt 1
 * und so weiter). Ein Benutzer erhlt dann schlielich als gesamt SpamScore
 * seinen schlechtesten.
 * 
 * @author Jens Bothur
 */
public class RatingSpamScoreCalculator implements ISingleSpamScoreCalculator {

	/**
	 * Eine Map in der der aktuelle Stand der Berechnung hinterlegt wird.
	 */
	private static Map<User, Double> _result;

	/**
	 * Diese Methode berechnet die BewertungsSpamScores fr alle Benutzers. Sie
	 * bentigt dazu eine Sammlung der Benutzer und eine Menge von allen Gruppen
	 * von bewertbaren Objekten. Das Ergebniss wird dann in einer Map geliefert,
	 * in der jeder Benutzer zu seinem Score zugeordnet wird.
	 * 
	 * @param users
	 *            Eine {@link Collection} von allen {@link User}n des Systems.
	 * @param objectGroups
	 *            Ein {@link Set} von allen {@link GroupOfRateableObjects} des
	 *            Systems.
	 * @return Eine {@link Map} welche alle Benutzer zu ihren zugerigen
	 *         Bewertungs-Spam-Scores als double zuordnet.
	 */
	public static Map<User, Double> calculateRatingSpamScore(
			Collection<User> users, Set<GroupOfRateableObjects> objectGroups) {
		_result = new LinkedHashMap<User, Double>();

		for (User user : users) {
			_result.put(user, 0.0);
		}

		for (GroupOfRateableObjects groupOfRateableObjects : objectGroups) {
			calculateAndInduceObjectSpecificRatingSpamScores(groupOfRateableObjects);
		}

		return _result;
	}

	/**
	 * Diese Methode berechnet den objekt gruppen spezifischen Spamming Score
	 * fr alle Benutzer die diese Objektgruppe bewertet haben. Nach Berechnung
	 * wird berprft of der berechnete Spamming-Score schlechter ist als der
	 * aktuell in {@link #_result} vermerkte. Ist dem so wird der neu berechnete
	 * eingetragen.
	 * 
	 * @param groupOfRateableObjects
	 *            Eine {@link GroupOfRateableObjects} fr die die spezifischen
	 *            Bewertungs-Spam-Scores berechnet werden sollen.
	 */
	private static void calculateAndInduceObjectSpecificRatingSpamScores(
			GroupOfRateableObjects groupOfRateableObjects) {
		Set<User> raters = groupOfRateableObjects.getRaters();
		Set<RateableObject> members = groupOfRateableObjects.getMembers();

		Map<User, Double> unnormalizedSpammingScores = new LinkedHashMap<User, Double>();
		double maximumUnnormalizedSpammingScore = 0.0;
		for (User rater : raters) {
			double unnormalizedSpammingScore = calculateUnnormalizedSpammingPartForUser(
					rater, members);
			unnormalizedSpammingScores.put(rater, unnormalizedSpammingScore);
			if (unnormalizedSpammingScore > maximumUnnormalizedSpammingScore) {
				maximumUnnormalizedSpammingScore = unnormalizedSpammingScore;
			}
		}

		for (User rater : raters) {
			double normalizedObjectSpecificSpammingScore = unnormalizedSpammingScores
					.get(rater) / maximumUnnormalizedSpammingScore;
			double normalizedTotalSpammingScore = _result.get(rater);
			if (normalizedObjectSpecificSpammingScore > normalizedTotalSpammingScore) {
				_result.put(rater, normalizedObjectSpecificSpammingScore);
			}
		}

	}

	/**
	 * Berechnet den unnormalisierten Spamming-Anteil eines Benutzers bezglich
	 * einer Gruppe von bewertbaren Objekten.
	 * 
	 * @param rater
	 *            Ein {@link User} welcher mindestens 2 Mitglieder von members
	 *            bewertet hat.
	 * @param members
	 *            Ein {@link Set} von {@link RateableObject} welche Mitglied
	 *            einer Objektgruppe sind, die der rater bewertet hat.
	 * @return Den unnormalisierten Spamming-Anteil des raters bezglich der
	 *         members als {@link Double}.
	 */
	private static double calculateUnnormalizedSpammingPartForUser(User rater,
			Set<RateableObject> members) {
		// find all reviews by this user
		List<Review> ratersReviews = new ArrayList<Review>();
		for (RateableObject rateableObject : members) {
			Review ratersReview = rateableObject.getReviewForReviewer(rater
					);
			if (ratersReview != null) {
				ratersReviews.add(ratersReview);
			}
		}
		// calculate similarity function sim
		int counter = 0;
		double total = 0.0;
		for (int i = 0; i < ratersReviews.size(); i++) {
			for (int j = i + 1; j < ratersReviews.size(); j++) {
				counter++;
				total += Math.abs(ratersReviews.get(i).getNormalizedRating()
						- ratersReviews.get(j).getNormalizedRating());
			}
		}
		double average = total / (double) counter;
		double sim = 1 - average;
		// calculate Spamming Part
		double spammingPart = Math.pow(ratersReviews.size(), 2) * sim;

		return spammingPart;

	}
}
