/**
 * Bachelorarbeit: Crowdsourcing- Systeme Manipulation und Resistenz
 * von Jens Bothur
 */
package jens.bothur.occt.domainobjects;

import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import jens.bothur.occt.domainvalues.AverageReview;
import jens.bothur.occt.domainvalues.Review;

/**
 * Diese Klasse reprsentiert ein bewertbares Objekt. Ein bewertbares Objekt ist
 * nicht viel mehr als ein Container fr alle IDs von Benutzern ({@link User})
 * die eine Bewertung ({@link Review}) fr diese Objekt abgegeben haben und fr
 * die durchschnittliche Bewertung die dieses Objekt hat. Eine ID wird auch
 * vergeben. Zuknftige bewertbare Objekte, wie z.B. Termine im OCCT, sollten
 * von diese Klasse erben und ihre speziellen Eigentschaften ergnzen.
 * 
 * @author Jens Bothur
 */
public class RateableObject {

	/**
	 * Die nchste noch nicht benutze Id.
	 */
	private static long _currentObjectId = 0;

	/**
	 * Gibt die nchste noch nicht benutze Id zurck und erhht den Counter.
	 * 
	 * @return die nchste noch nicht benutzte Id als long.
	 */
	private static long nextId() {
		return _currentObjectId++;
	}

	/**
	 * Die ID dieses bewertbaren Objektes.
	 */
	private final long ID;

	/**
	 * Eine {@link Map} worin alle IDs von Benutzer welche dieses Objekt
	 * bewertet haben und die entsprechende Bewertung abgelegt werden.
	 */
	private final Map<User, Review> REVIEWS;

	/**
	 * Eine Map worin alle IDs von Benutzer, welche dieses Objekt bewertet
	 * haben, dem entsprechenden zeitpunkt zugeordnet werden.
	 */
	private final Map<User, Date> DATES;

	/**
	 * Die aktuelle durchschnittliche Bewertung dieses bewertbaren Objekts als
	 * {@link AverageReview}.
	 */
	private AverageReview _averageReview;

	/**
	 * Das Datum an dem frhestens eine Bewertung mglich ist.
	 */
	private Date _earliestReview;

	/**
	 * Konstruktor fr ein neues bewertbares Objekt. Die ID diese Objekts wird
	 * automatisch bestimmt. Die durchschnittlcihe Bewertung liegt zu Beginn bei
	 * 3.0, ebenso gibt es zu Beginn keine Bewertungen fr diese bewertbare
	 * Objekt. Das Datum fr die frhest mgliche Bewertung wird auf den
	 * 01.01.0000 00:00:01 gelegt
	 */
	@SuppressWarnings("deprecation")
	public RateableObject() {
		this(new Date(-1900, 1, 1, 0, 0, 1));
	}

	/**
	 * Konstruktor fr ein neues bewertbares Objekt. Die ID diese Objekts wird
	 * automatisch bestimmt. Die durchschnittlcihe Bewertung liegt zu Beginn bei
	 * 3.0, ebenso gibt es zu Beginn keine Bewertungen fr diese bewertbare
	 * Objekt. Das Datum fr den Zeitpunkt der frhest mglichen Bewertung wird
	 * bergeben.
	 * 
	 * @param earliestReview
	 *            Ein {@link Date} fr den Zeitpunkt der frhest mglichen
	 *            bewertung.
	 */
	public RateableObject(Date earliestReview) {
		ID = nextId();
		REVIEWS = new HashMap<User, Review>();
		DATES = new HashMap<User, Date>();
		_averageReview = AverageReview.getAverageReviewByIntTimes10(30);
		_earliestReview = earliestReview;
	}

	/**
	 * Gibt die aktuelle durchschnittliche Bewertung dieses bewertbaren Objekts
	 * als {@link AverageReview} zurck.
	 * 
	 * @return Die aktuelle durchschnittliche Bewertung dieses bewertbaren
	 *         Objekts als {@link AverageReview}.
	 */
	public AverageReview getAverageReview() {
		return _averageReview;
	}

	/**
	 * Setz Die aktuelle durchschnittliche Bewertung dieses bewertbaren Objekts
	 * neu.
	 * 
	 * @param averageReview
	 *            Die neue gewnschte durchschnittliche Bewertung als
	 *            {@link AverageReview}.
	 */
	public void setAverageReview(AverageReview averageReview) {
		_averageReview = averageReview;
	}

	/**
	 * Gibt die ID dieses bewertbaren Objektes zurck.
	 * 
	 * @return Die ID dieses bewertbaren Objektes als long.
	 */
	public long getID() {
		return ID;
	}

	/**
	 * Gibt die Menge aller Bewerter von diesem bewertbaren Objekt zurck.
	 * 
	 * @return Ein {@link Set} von {@link User}s welche dieses
	 *         {@link RateableObjectTest} bewertet haben.
	 */
	public Set<User> getReviewers() {
		return REVIEWS.keySet();
	}

	/**
	 * Gibt zu einem spezifizierten Benutzer die entsprechende Bewertung zurck.
	 * 
	 * @param user
	 *            Der {@link User} zu dem {@link Review} zurckzugeben ist.
	 * @return Das entsprechende {@link Review} zum user, <code>null</code> wenn
	 *         dieser Benutzer dieses bewertbare Objekt nicht bewertet hat.
	 */
	public Review getReviewForReviewer(User user) {
		return REVIEWS.get(user);
	}

	/**
	 * Gibt eine Sammlung aller Daten zurck an denen eine Bewertung fr dieses
	 * Objekt abgegeben wurde.
	 * 
	 * @return Eine {@link Collection} aller {@link Date}s an denen eine
	 *         Bewertung fr dieses Objekt abgegeben wurde.
	 */
	public Collection<Date> getRatingDates() {
		return DATES.values();
	}

	/**
	 * Gibt zu einer spezifizierten ID das Datum der entsprechenden Bewertung
	 * zurck.
	 * 
	 * @param user
	 *            Der {@link User} zu dem das {@link Date} zurckzugeben ist.
	 * @return Das entsprechende {@link Date} zur Bewertung durch den Benutzer ,
	 *         <code>null</code> wenn dieser Nenutzer dieses Objekt nicht
	 *         bewertet hat.
	 */
	public Date getDateForReviewer(User user) {
		return DATES.get(user);
	}

	/**
	 * Gibt das Datum fr den Zeitpunkt der frhest mglichen bewertung zurck.
	 * 
	 * @return Ein {@link Date} fr den Zeitpunkt der frhest mglichen
	 *         bewertung.
	 */
	public Date getEarliestPossibleReviewDate() {
		return _earliestReview;
	}

	/**
	 * Fgt diesem bewertbaren Objekt eine neue Bewertung hinzu. Sollte der
	 * bergebene Benutzer dieses Objekt bereits bewertet haben, wird
	 * <code>false</code> zurckgegeben.
	 * 
	 * @param user
	 *            Ein {@link User}, welcher dieses Objekt gerade bewerten
	 *            mchte.
	 * @param review
	 *            Das zugehrige {@link Review} welches der Benutzer gerade
	 *            abgeben mchte.
	 * @param date
	 *            Das {@link Date} an welchen diese Bewertung abgegeben wird.
	 * @return <code>true</code> wenn die Bewertung erfolgreich eingetragen
	 *         wurde, <code>false</code> wenn ein Benutzer mit diese ID bereits
	 *         eine Bewertung abgegeben hat.
	 */
	public boolean rate(User user, Review review, Date date) {
		if (date.before(_earliestReview)) {
			throw new IllegalArgumentException(
					"Illegal rating date. The rating was supposed to take place at "
							+ date
							+ " but the earliest possible review for this rateable object is "
							+ _earliestReview);

		}
		boolean result = false;
		if (!REVIEWS.keySet().contains(user)) {
			REVIEWS.put(user, review);
			DATES.put(user, date);
			result = true;
		}

		return result;
	}


	/**
	 * {@inheritDoc}
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + (int) (ID ^ (ID >>> 32));
		return result;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		RateableObject other = (RateableObject) obj;
		if (ID != other.ID)
			return false;
		return true;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return "RateableObject [ID=" + ID + "]";
	}

}
