/**
 * Bachelorarbeit: Crowdsourcing- Systeme Manipulation und Resistenz
 * von Jens Bothur
 */
package jens.bothur.occt.domainobjects;

import java.util.LinkedHashSet;
import java.util.Set;

import jens.bothur.occt.domainvalues.Review;

/**
 * Diese Klasse reprsentiert eine Bewerter Gruppe. Diese Gruppe besteht aus
 * Benutzern ( {@link User} ) des System. Eine Gruppe muss mindestens die Gre
 * {@link #MINIMUM_SIZE} haben und mindestens bei
 * {@link #MINIMUM_COLLABORATION_COUNT} bewertbaren Objekten zusammengearbeitet
 * haben.
 * 
 * @author Jens Bothur
 */
public class ReviewerGroup {

	/**
	 * Die globale minimale Gre einer Bewerter-Gruppe.
	 */
	public final int MINIMUM_SIZE = 2;

	/**
	 * Die globale minimale Anzahl an bewertbaren Objekten die alle in dieser
	 * Gruppe bewertet haben mssen.
	 */
	public final int MINIMUM_COLLABORATION_COUNT = 3;

	/**
	 * Eine Menge mit allen Mitgliedern dieser Gruppe.
	 */
	private final Set<User> _members;

	/**
	 * Eine Menge mit allen bewertbaren Objekten, bei denen diese Gruppe
	 * zusammengearbeitet hat.
	 */
	private final Set<RateableObject> _collaboratedObjects;

	/**
	 * Konstruktor fr Bewerter-Gruppen. Die Mitglieder und die gemeinsam
	 * bewerteten Objekte werden bergeben.
	 * 
	 * @param members
	 *            Ein {@link Set} mit allen Mitgliedern dieser Gruppe.
	 * @param collaboratedObjects
	 *            Ein {@link Set} mit allen gemeinsam bewerteten Objekten dieser
	 *            Gruppe.
	 * 
	 * @require Jedes Mitglied muss jedes gemeinsam bewertete Objekt bewertet
	 *          haben.
	 * @require members.size() >= {@link #MINIMUM_SIZE}
	 * @require collaboratedObjects.size() >=
	 *          {@link #MINIMUM_COLLABORATION_COUNT}
	 */
	public ReviewerGroup(Set<User> members,
			Set<RateableObject> collaboratedObjects) {
		assert members.size() >= MINIMUM_SIZE;
		assert collaboratedObjects.size() >= MINIMUM_COLLABORATION_COUNT;
		for (RateableObject rateableObject : collaboratedObjects) {
			for (User user : members) {
				Review reviewForReviewerID = rateableObject
						.getReviewForReviewer(user);
				assert reviewForReviewerID != null;
			}

		}
		_members = members;
		_collaboratedObjects = collaboratedObjects;
	}

	/**
	 * Gibt ein {@link Set} mit allen Mitgliedern dieser Gruppe zurck.
	 * 
	 * @return Ein {@link Set} mit allen Mitgliedern dieser Gruppe.
	 */
	public Set<User> getMembers() {
		return new LinkedHashSet<User>(_members);
	}

	/**
	 * Gibt ein {@link Set} mit allen gemeinsam bewerteten Objekten dieser
	 * Gruppe zurck.
	 * 
	 * @return Ein {@link Set} mit allen gemeinsam bewerteten Objekten dieser
	 *         Gruppe.
	 */

	public Set<RateableObject> getCollaboratedObjects() {
		return new LinkedHashSet<RateableObject>(_collaboratedObjects);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime
				* result
				+ ((_collaboratedObjects == null) ? 0 : _collaboratedObjects
						.hashCode());
		result = prime * result
				+ ((_members == null) ? 0 : _members.hashCode());
		return result;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		ReviewerGroup other = (ReviewerGroup) obj;
		if (!_collaboratedObjects.equals(other._collaboratedObjects))
			return false;
		if (!_members.equals(other._members))
			return false;
		return true;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return "ReviewerGroup [_members=" + _members + "] + [_objects="
				+ _collaboratedObjects + "]";
	}
}
