/**
 * Bachelorarbeit: Crowdsourcing- Systeme Manipulation und Resistenz
 * von Jens Bothur
 */
package jens.bothur.occt.domainvalues;

import java.util.LinkedHashSet;
import java.util.Set;

/**
 * Diese Klasse reprsentiert ein Bewertung fr ein bewertbares Objekt. Eine
 * Bewertung nimmt einen ganzzahligen Wert zwischen 1 und 5 an. Somit kann ein
 * Termin im OCCT mit 1-5 "Sternen" bewertet werden.
 * 
 * @author Jens Bothur
 */
public class Review {

	/**
	 * Eine Menge welche alle erlaubten Bewertungen beeinhaltet.
	 */
	@SuppressWarnings("serial")
	private static final Set<Integer> ALLOWED_RATINGS = new LinkedHashSet<Integer>() {
		{
			add(1);
			add(2);
			add(3);
			add(4);
			add(5);
		}
	};

	/**
	 * Die "ein-Sterne" Bewertung.
	 */
	public static final Review ONE_STAR_REVIEW = new Review(1);

	/**
	 * Die "zwei-Sterne" Bewertung.
	 */
	public static final Review TWO_STAR_REVIEW = new Review(2);

	/**
	 * Die "drei-Sterne" Bewertung.
	 */
	public static final Review THREE_STAR_REVIEW = new Review(3);

	/**
	 * Die "vier-Sterne" Bewertung.
	 */
	public static final Review FOUR_STAR_REVIEW = new Review(4);

	/**
	 * Die "fnf-Sterne" Bewertung.
	 */
	public static final Review FIVE_STAR_REVIEW = new Review(5);

	/**
	 * Ein ganzzahliger Wert zwischen 1 und 5.
	 */
	private int _rating;

	/**
	 * Privater Constructor um Bewertungen zu erzeugen.
	 * 
	 * @param rating
	 *            Die vergebene Bewertung als int.
	 * 
	 * @require {@link #ALLOWED_RATINGS}.contains(rating)
	 */
	private Review(int rating) {
		if (!ALLOWED_RATINGS.contains(rating)) {
			throw new IllegalArgumentException(rating
					+ " wasn't an allowed rating. Allowed ratings are: "
					+ ALLOWED_RATINGS);
		}

		_rating = rating;
	}

	/**
	 * Gibt die vergebene Bewertung dieses {@link Review}s wieder als int.
	 * 
	 * @return die vergebene Bewertung dieses {@link Review}s wieder als int.
	 * 
	 * @ensure {@link #ALLOWED_RATINGS}.contains(returned value).
	 */
	public int getRating() {
		return _rating;
	}

	/**
	 * Gibt die vergebene Bewertung dieses {@link Review}s als double
	 * normalisiert auf das Intervall [0;1] wieder.
	 * 
	 * ACHTUNG: Diese Methode muss angepasst werden, sollte
	 * {@link #ALLOWED_RATINGS} verndert werden.
	 * 
	 * @return Gibt die vergebene Bewertung dieses {@link Review}s als double
	 *         normalisiert auf das Intervall [0;1] wieder.
	 */
	public double getNormalizedRating() {
		return (_rating - 1) / 4.0;
	}
}
