/**
 * Bachelorarbeit: Crowdsourcing- Systeme Manipulation und Resistenz
 * von Jens Bothur
 */
package jens.bothur.occt.automatons;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.util.Date;

import jens.bothur.occt.domainobjects.RateableObject;
import jens.bothur.occt.domainobjects.User;
import jens.bothur.occt.domainvalues.AverageReview;
import jens.bothur.occt.domainvalues.ConfidenceLevel;
import jens.bothur.occt.domainvalues.Review;
import jens.bothur.occt.services.DummyRateableObjectService;
import jens.bothur.occt.services.DummyUserService;
import jens.bothur.occt.services.IRateableObjectService;
import jens.bothur.occt.services.IUserService;

import org.junit.Before;
import org.junit.Test;

/**
 * Test fr die Klasse {@link AverageRatingCalculator}
 * 
 * @author Jens Bothur
 */
public class AverageRatingCalculatorTest {

	private AverageRatingCalculator _instance;

	private IUserService _userService;
	private IRateableObjectService _objectService;

	private final RateableObject _object1 = new RateableObject();
	private final RateableObject _object2 = new RateableObject();
	private final RateableObject _object3 = new RateableObject();

	private final User _user1 = new User("a");
	private final User _user2 = new User("b");
	private final User _user3 = new User("c");

	/**
	 * @throws java.lang.Exception
	 */
	@Before
	public void setUp() throws Exception {
		_userService = new DummyUserService();
		_userService.registerUser(_user1);
		_userService.registerUser(_user2);
		_userService.registerUser(_user3);

		_objectService = new DummyRateableObjectService();
		_objectService.registerRateableObject(_object1);
		_objectService.registerRateableObject(_object2);
		_objectService.registerRateableObject(_object3);

		_instance = new AverageRatingCalculator(_objectService);
	}

	@Test
	public void testCalculateRatings() {
		_instance.calculateAverageRatings();
		assertEquals(AverageReview.getAverageReviewByIntTimes10(30),
				_object1.getAverageReview());
		assertEquals(AverageReview.getAverageReviewByIntTimes10(30),
				_object2.getAverageReview());
		assertEquals(AverageReview.getAverageReviewByIntTimes10(30),
				_object3.getAverageReview());

		assertTrue(_object1.rate(_user1, Review.ONE_STAR_REVIEW, new Date()));
		assertTrue(_object2.rate(_user1, Review.TWO_STAR_REVIEW, new Date()));
		assertTrue(_object3.rate(_user1, Review.THREE_STAR_REVIEW, new Date()));
		_instance.calculateAverageRatings();
		assertEquals(AverageReview.getAverageReviewByIntTimes10(10),
				_object1.getAverageReview());
		assertEquals(AverageReview.getAverageReviewByIntTimes10(20),
				_object2.getAverageReview());
		assertEquals(AverageReview.getAverageReviewByIntTimes10(30),
				_object3.getAverageReview());

		assertTrue(_object1.rate(_user2, Review.THREE_STAR_REVIEW, new Date()));
		assertTrue(_object2.rate(_user2, Review.TWO_STAR_REVIEW, new Date()));
		assertTrue(_object3.rate(_user2, Review.ONE_STAR_REVIEW, new Date()));
		// alle objekte sollten nun eine durchschnittliche bewertung von 2
		// erhalten.
		_instance.calculateAverageRatings();
		assertEquals(AverageReview.getAverageReviewByIntTimes10(20),
				_object1.getAverageReview());
		assertEquals(AverageReview.getAverageReviewByIntTimes10(20),
				_object2.getAverageReview());
		assertEquals(AverageReview.getAverageReviewByIntTimes10(20),
				_object3.getAverageReview());

		_user3.setConfidenceLevel(ConfidenceLevel
				.getConfidenceLevelByIntTimes10(80));
		assertTrue(_object1.rate(_user3, Review.ONE_STAR_REVIEW, new Date()));
		assertTrue(_object2.rate(_user3, Review.THREE_STAR_REVIEW, new Date()));
		assertTrue(_object3.rate(_user3, Review.FIVE_STAR_REVIEW, new Date()));
		/*
		 * objekt 1 sollte nun ein rating von 1,2 (1*1 + 1*3 + 8*1) erhalten.
		 * objekt 2 sollte nun ein rating von 2,8 (1*2 + 1*2 + 8*3) erhalten.
		 * objekt 3 sollte nun ein rating von 4,4 (1*3 + 1*1 + 8*5) erhalten.
		 */
		_instance.calculateAverageRatings();
		assertEquals(AverageReview.getAverageReviewByIntTimes10(12),
				_object1.getAverageReview());
		assertEquals(AverageReview.getAverageReviewByIntTimes10(28),
				_object2.getAverageReview());
		assertEquals(AverageReview.getAverageReviewByIntTimes10(44),
				_object3.getAverageReview());

		_user1.setConfidenceLevel(ConfidenceLevel
				.getConfidenceLevelByIntTimes10(55));
		_user2.setConfidenceLevel(ConfidenceLevel
				.getConfidenceLevelByIntTimes10(0));
		_user3.setConfidenceLevel(ConfidenceLevel
				.getConfidenceLevelByIntTimes10(45));
		/*
		 * objekt 1 sollte nun ein rating von 1 (5,5*1 + 0*3 + 4,5*1) erhalten.
		 * objekt 2 sollte nun ein rating von 2,4 (5,5*2 + 0*2 + 4,5*3)
		 * erhalten.
		 * 
		 * objekt 3 sollte nun ein rating von 3,9 (5,5*3 + 0*1 + 4,5*5)
		 * erhalten.
		 */
		_instance.calculateAverageRatings();
		assertEquals(AverageReview.getAverageReviewByIntTimes10(10),
				_object1.getAverageReview());
		assertEquals(AverageReview.getAverageReviewByIntTimes10(24),
				_object2.getAverageReview());
		assertEquals(AverageReview.getAverageReviewByIntTimes10(39),
				_object3.getAverageReview());
	}

}
