/**
 *  Serponix is an arcade game in focus to multiplayer based on the classic game Snake.
 *  Copyright (C) 2010 - 2011  Daniel Vala
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License,
 *  or  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *  If you have any question do not hesitate to contact author
 *  on e-mail address: danielvala42@gmail.com
 */
package com.serponix.abilities;

import com.serponix.game.ModelFun;
import com.serponix.game.Snake;

/**
 * Vlastnost obsahuje svojí šanci na vygenerování se.
 * Dále má uloženou svojí pravděpodobnost vygenerování se vůči všem ostatním bonusům od 0 do 1.
 *
 * @author Daniel Vala
 */
public abstract class Ability {

	// Šance vygenerování bonusu od 0 do rozsahu int.
	// Čím nižší číslo je, tím nižší je jeho šance se vygenerovat.
	// 0 znamená, že se vlastnost nikdy nevygeneruje.
	private int sance;
	// Pravděpodobnost vygenerování zrovna tohoto bonusu od 0 do 1
	private double pravdepodobnost;

	/**
	 * Vytvoří novou vlastonst s danou šancí.
	 *
	 * @param sance Šance bonusu na jeho vygenerování se od 0 do rozsahu int.
	 */
	public Ability(int sance) {
		this.sance = sance;
	}

	/**
	 * Metoda se pokusí získat pro hada tuto vlastnost.
	 *
	 * @param model Model hry.
	 * @param had   Had, který získá vlastnost.
	 * @return True při úspěchu. False, pokud had vlastnost nezískal.
	 */
	public abstract boolean proved(ModelFun model, Snake had);

	/**
	 * Vrátí pravděpodobnost vygenerování právě této vlastnosti vůči všem ostatním.
	 *
	 * @return Pravděpodobnost od 0 do 1.
	 */
	public double getPravddepodobnost() {
		return pravdepodobnost;
	}

	/**
	 * Nastaví pravděpodobnost vygenerování se této vlastnosti vůči všem ostatním bonusům.
	 *
	 * @param sanceVProcentech Od 0 do 1.
	 */
	public void setPravdepodobnost(double sanceVProcentech) {
		if (sanceVProcentech < 0 || sanceVProcentech > 1) {
			throw new IllegalArgumentException("Pravděpodobnost musí být v rozsahu od 0 do 1.");
		}
		this.pravdepodobnost = sanceVProcentech;
	}

	/**
	 * Vrátí šanci bonusu, že se vygeneruje.
	 *
	 * @return Šance bonusu na vygenerování se od 0 do rozsahu int.
	 */
	public int getSance() {
		return sance;
	}
}
