/**
 *  Serponix is an arcade game in focus to multiplayer based on the classic game Snake.
 *  Copyright (C) 2010 - 2011  Daniel Vala
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License,
 *  or  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *  If you have any question do not hesitate to contact author
 *  on e-mail address: danielvala42@gmail.com
 */
package com.serponix.game.objektyNaPlose;

import com.serponix.game.Direction;
import com.serponix.game.Snake;
import java.awt.AlphaComposite;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Polygon;

/**
 * Článek hada obsahuje informace o tom, jakému hadovi patří
 * a kolikátý jeho článek je.
 * Poskytuje také metody pro vykreslení různých typů článků.
 *
 * @author Daniel Vala
 */
public class Clanek extends VecNaPolicku {

	private Snake had;

	public Clanek(Snake had, int x, int y) {
		super(x, y);
		this.had = had;
	}

	public Clanek(Snake had) {
		this.had = had;
	}

	/**
	 * Vykreslí článek hada na dané souřadnice.
	 *
	 * @param g2d      Grafický objekt, kterým se bude kreslit.
	 * @param x        Souřadnice článku X.
	 * @param y        Souřadnice článku Y.
	 * @param tloustka Tlouštka článku.
	 */
	public static void paint(Graphics2D g2d, int x, int y, int tloustka) {
		x *= tloustka;
		y *= tloustka;
		g2d.fill3DRect(x, y, tloustka, tloustka, true);
	}

	/**
	 * Vykreslí článek odolného hada na dané souřadnice.
	 *
	 * @param g2d      Grafický objekt, kterým se bude kreslit.
	 * @param x        Souřadnice článku X.
	 * @param y        Souřadnice článku Y.
	 * @param tloustka Tlouštka článku.
	 */
	public static void paintOdolny(Graphics2D g2d, int x, int y, int tloustka) {
		x *= tloustka;
		y *= tloustka;
		g2d.draw3DRect(x, y, tloustka, tloustka, false);
	}

	/**
	 * Vykreslí článek ducha hada na dané souřadnice, což jsou malé čtverečky.
	 *
	 * @param g2d      Grafický objekt, kterým se bude kreslit.
	 * @param x        Souřadnice článku X.
	 * @param y        Souřadnice článku Y.
	 * @param tloustka Tlouštka článku.
	 */
	public static void paintDuch(Graphics2D g2d, int x, int y, int tloustka) {
		x *= tloustka;
		y *= tloustka;
		g2d.fillRect(x + (tloustka / 3), y + (tloustka / 3), (tloustka / 2), (tloustka / 2));
	}

	/**
	 * Nastaví předaný grafický objekt na průhlednost, při které se kreslí neviditelný článek hada.
	 *
	 * @param g2d Grafický objekt, který se nastaví na průhlednost.
	 */
	public static void NastavNeviditelneKresleni(Graphics2D g2d) {
		g2d.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, 0.3f));
	}

	/**
	 * Vykreslí hlavu hada do zadaných souřadnic podle zadaného směru a tlouštky.
	 * Je třeba předem nastavit barvu článku.
	 *
	 * @param g2d
	 * @param x        Souřadnice hlavy x.
	 * @param y        Souřadnice hlavy y.
	 * @param tloustka Tlouštka hlavy.
	 * @param smer     Směr článku, podle kterého bude hlava natočena. Při zadání směru NIKAM nebo null nevykreslí nic.
	 */
	public static void paintHlava(Graphics2D g2d, int x, int y, int tloustka, Direction smer) {
		x *= tloustka;
		y *= tloustka;
		int velikostOka = tloustka / 10;
		int posunOkaDoprostred = tloustka / 5;
		int posunOkaPodelHavy = tloustka / 10;

		Color puvodniBarva = g2d.getColor();
		switch (smer) {
			case UP:
				g2d.fillArc(x, y + (tloustka / 2), tloustka, tloustka, 0, 180);
				g2d.setPaint(Color.BLACK);
				g2d.fillOval(x + posunOkaDoprostred, y + (tloustka - posunOkaPodelHavy - velikostOka), velikostOka, velikostOka);
				g2d.fillOval(x + (tloustka - posunOkaDoprostred - velikostOka), y + (tloustka - posunOkaPodelHavy - velikostOka), velikostOka, velikostOka);
				break;
			case DOWN:
				g2d.fillArc(x, y - (tloustka / 2), tloustka, tloustka, 0, -180);
				g2d.setPaint(Color.BLACK);
				g2d.fillOval(x + posunOkaDoprostred, y + posunOkaPodelHavy, velikostOka, velikostOka);
				g2d.fillOval(x + (tloustka - posunOkaDoprostred - velikostOka), y + posunOkaPodelHavy, velikostOka, velikostOka);
				break;
			case LEFT:
				g2d.fillArc(x + (tloustka / 2), y, tloustka, tloustka, 90, 180);
				g2d.setPaint(Color.BLACK);
				g2d.fillOval(x + (tloustka - posunOkaPodelHavy - velikostOka), y + posunOkaDoprostred, velikostOka, velikostOka);
				g2d.fillOval(x + (tloustka - posunOkaPodelHavy - velikostOka), y + (tloustka - posunOkaDoprostred - velikostOka), velikostOka, velikostOka);
				break;
			case RIGHT:
				// kreslení hlavy
				g2d.fillArc(x - (tloustka / 2), y, tloustka, tloustka, 90, -180);
				// kreslení očí
				g2d.setPaint(Color.BLACK);
				// horní oko
				g2d.fillOval(x + posunOkaPodelHavy, y + posunOkaDoprostred, velikostOka, velikostOka);
				// dolní oko
				g2d.fillOval(x + posunOkaPodelHavy, y + (tloustka - posunOkaDoprostred - velikostOka), velikostOka, velikostOka);
				break;
		}
		// nastavení barvy tak, jak byla před voláním této metody
		g2d.setPaint(puvodniBarva);
	}

	/**
	 * Vykreslí hlavu ve tvaru tanku (do špičky) do zadaných souřadnic podle zadaného směru a tlouštky.
	 * Je třeba předem nastavit barvu článku.
	 *
	 * @param g2d
	 * @param x        Souřadnice hlavy x.
	 * @param y        Souřadnice hlavy y.
	 * @param tloustka Tlouštka hlavy.
	 * @param smer     Směr článku, podle kterého bude hlava natočena. Při zadání směru NIKAM nebo null nevykreslí nic.
	 */
	public static void paintHlavaTank(Graphics2D g2d, int x, int y, int tloustka, Direction smer) {
		x *= tloustka;
		y *= tloustka;
		Polygon hlava = null;
		int[] xPoints = null;
		int[] yPoints = null;
		switch (smer) {
			case UP:
				xPoints = new int[] { x, x + tloustka, x + (tloustka / 2) };
				yPoints = new int[] { y + tloustka, y + tloustka, y };
				break;
			case DOWN:
				xPoints = new int[] { x, x + tloustka, x + (tloustka / 2) };
				yPoints = new int[] { y, y, y + tloustka };
				break;
			case RIGHT:
				xPoints = new int[] { x, x + tloustka, x };
				yPoints = new int[] { y, y + (tloustka / 2), y + tloustka };
				break;
			case LEFT:
				xPoints = new int[] { x + tloustka, x, x + tloustka };
				yPoints = new int[] { y, y + (tloustka / 2), y + tloustka };
				break;
			default:
				return;
		}
		hlava = new Polygon(xPoints, yPoints, 3);
		g2d.fillPolygon(hlava);
		g2d.draw(hlava);
	}

	public Snake getHad() {
		return had;
	}
}
