/**
 *  Serponix is an arcade game in focus to multiplayer based on the classic game Snake.
 *  Copyright (C) 2010 - 2011  Daniel Vala
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License,
 *  or  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *  If you have any question do not hesitate to contact author
 *  on e-mail address: danielvala42@gmail.com
 */
package com.serponix.game.projectiles;

import com.serponix.game.Direction;
import com.serponix.game.GameModel;
import com.serponix.game.Snake;
import com.serponix.game.objektyNaPlose.Clanek;
import com.serponix.game.objektyNaPlose.VecNaPolicku;
import com.serponix.game.objektyNaPlose.Zed;
import com.serponix.net.Informace;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.MediaTracker;
import java.awt.Panel;
import java.awt.Toolkit;
import java.awt.image.BufferedImage;
import javax.swing.JPanel;

/**
 * Rocket is heavy projectile, which do not destroy itself when hit snake.
 * Snake lose half of his tail and tail is in this case collision cell.
 * When rocket hits wall, both wall and rocket are destroyed.
 *
 * @author Daniel Vala
 */
public class Rocket extends Projectile {

	/**
	 * Images of rocket with right, left, up and down direction.
	 */
	private static final Image rocketR;
	private static final Image rocketL;
	private static final Image rocketU;
	private static final Image rocketD;

	/**
	 * Loading image of rocket with right direction
	 * and after rotating saving 4 images with all rocket direction.
	 */
	static {
		JPanel panel = new JPanel();
		MediaTracker mt = new MediaTracker(panel);
		rocketR = Toolkit.getDefaultToolkit().getImage(Panel.class.getResource("/images/rocket.png"));
		mt.addImage(rocketR, 0);
		try {
			mt.waitForID(0);
		} catch (InterruptedException ex) {
			System.err.println("Error in loading rocket image. " + ex.getMessage());
		}

		int width = rocketR.getWidth(null);
		int height = rocketR.getHeight(null);

		rocketL = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
		Graphics2D g = (Graphics2D) rocketL.getGraphics();
		g.rotate(Math.PI, rocketL.getWidth(null) / 2, rocketL.getHeight(null) / 2);
		g.drawImage(rocketR, 0, 0, null);

		rocketU = new BufferedImage(height, width, BufferedImage.TYPE_INT_ARGB);
		g = (Graphics2D) rocketU.getGraphics();
		g.rotate(3 * Math.PI / 2, rocketU.getWidth(null) / 2, rocketU.getHeight(null) / 2);
		int difference = (rocketU.getWidth(null) - rocketU.getHeight(null)) / 2;
		g.drawImage(rocketR, difference, -difference, null);

		rocketD = new BufferedImage(height, width, BufferedImage.TYPE_INT_ARGB);
		g = (Graphics2D) rocketD.getGraphics();
		g.rotate(Math.PI / 2, rocketD.getWidth(null) / 2, rocketD.getHeight(null) / 2);
		difference = (rocketD.getWidth(null) - rocketD.getHeight(null)) / 2;
		g.drawImage(rocketR, difference, -difference, null);
	}

	public Rocket(GameModel model, Snake snake, int x, int y, Direction direction, int speed) {
		super(model, snake, x, y, direction, speed);
	}

	public static void paint(Graphics g, int x, int y, Direction smer) {
		x *= 10;
		y *= 10;
		switch (smer) {
			case UP:
				g.drawImage(rocketU, x - 3, y, 16, 30, null);
				break;
			case DOWN:
				g.drawImage(rocketD, x - 3, y, 16, 30, null);
				break;
			case LEFT:
				g.drawImage(rocketL, x, y - 3, 30, 16, null);
				break;
			case RIGHT:
				g.drawImage(rocketR, x, y - 3, 30, 16, null);
				break;
		}
	}

	@Override
	public String getInstrukceProKresleni() {
		return Informace.RAKETA + Informace.ODDELOVAC3 + getX() + Informace.ODDELOVAC3 + getY() + Informace.ODDELOVAC3 + 0 + Informace.ODDELOVAC3 + 0 + Informace.ODDELOVAC3 + Direction.prevedSmerNaCislo(smer) + Informace.ODDELOVAC2;
	}

	@Override
	public void znicSe() {
		snake.nabijRaketu();
		odmapuj(model);
	}

	/**
	 * Zkontroluje, zda střela nenarazila do hada nebo jiné překážky na mapě.
	 * Při srážce s hadem mu usekne polovinu ocasu.
	 * Při srážce s překážkou mapy při dostatečné síle rakety jí odstraní, ale zmenší se síla rakety o 1.
	 * Při síle rakety 0 raketa zaniká.
	 * Dále zkontroluje kolizi se stěnami herní plochy.
	 *
	 * @return <tt>true</tt> pokud byla raketa zničena. <tt>False</tt> pokud raketa letí dále.
	 * Může vrátit false i v případě nárazu, pokud to raketa přežije.
	 */
	@Override
	protected boolean kolize(int x, int y) {
		VecNaPolicku kolizniObjekt = model.getObjektNaHerniPlose(x, y);
		if (kolizniObjekt != null) {
			if (kolizniObjekt instanceof Clanek) {
				Clanek clanek = (Clanek) kolizniObjekt;
				Snake zasazenyHad = clanek.getHad();
				zasazenyHad.usekniEfektneOcas(x, y);
				return false;
			} else if (kolizniObjekt instanceof Zed) {
				Zed zed = (Zed) kolizniObjekt;
				model.getMapa().smazZed(model, zed);
				// poslání informace o zničení zdi všem klientům
				model.posliDataKlientum(Informace.POLICKO + Informace.ODDELOVAC1 + Informace.SMAZ_ZED + Informace.ODDELOVAC3 + zed.x + Informace.ODDELOVAC3 + zed.y + Informace.ODDELOVAC2);
				znicSe();
				return true;
			}
		}

		// kontrola krajních stěn
		if (!model.isVHerniMape(getX(), getY())) {
			znicSe();
			return true;
		}
		return false;
	}

	@Override
	public void draw(Graphics g) {
		paint(g, getX(), getY(), smer);
	}
}
