/**
 *  Serponix is an arcade game in focus to multiplayer based on the classic game Snake.
 *  Copyright (C) 2010 - 2011  Daniel Vala
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License,
 *  or  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *  If you have any question do not hesitate to contact author
 *  on e-mail address: danielvala42@gmail.com
 */
package com.serponix.gui.lobby;

import com.serponix.game.GameModeEnum;
import com.serponix.gui.MainPanel;
import com.serponix.net.Informace;
import com.serponix.net.Net;
import com.serponix.settings.GameSettings;
import java.awt.Dimension;
import java.util.Observable;

/**
 * Model obsahuje všechny společné parametry lobby.
 * Poskytuje metody pro poslání dat ostatním klientům,
 * přesun z lobby zpět do menu a nastavení parametrů hry.
 *
 * @author Daniel Vala
 */
public abstract class ModelLobby extends Observable {

	private MainPanel okno;
	private Net net;
	private int rychlostHry;
	private String nazevMapy;
	private GameModeEnum gameMode;
	private String[] jmenaHracu;
	private int sirkaHerniPlochy;
	private int vyskaHerniPlochy;

	public ModelLobby(MainPanel okno) {
		this.okno = okno;
		jmenaHracu = new String[4];
		for (int i = 0; i < jmenaHracu.length; i++) {
			jmenaHracu[i] = GameSettings.getInstance().getNameOfGivenPlayer(i);
		}
	}

	public void setNet(Net net) {
		this.net = net;
	}

	public void konec() {
		net.odpojSe();
		okno.displayMenu();
	}

	public void posliData(String data) {
		net.poslatData(data);
	}

	public String getJmenoHrace(int cisloHrace) {
		return jmenaHracu[cisloHrace];
	}

	public void setJmenoHrace(int cisloHrace, String jmenoHrace) {
		jmenaHracu[cisloHrace] = jmenoHrace;
	}

	public int getSirkaHerniPlochy() {
		return sirkaHerniPlochy;
	}

	public void setSirkaHerniPlochy(int sirkaHerniPlochy) {
		this.sirkaHerniPlochy = sirkaHerniPlochy;
	}

	public int getVyskaHerniPlochy() {
		return vyskaHerniPlochy;
	}

	public void setVyskaHerniPlochy(int vyskaHerniPlochy) {
		this.vyskaHerniPlochy = vyskaHerniPlochy;
	}

	public Dimension getVelikostHerniPlochy() {
		return new Dimension(sirkaHerniPlochy - 1, vyskaHerniPlochy - 1);
	}

	/**
	 * Get game mode which is currently set.
	 *
	 * @return Current game mode.
	 */
	public GameModeEnum getGameMode() {
		return gameMode;
	}

	/**
	 * Sets new game mode.
	 *
	 * @param gameMode New game mode to be set.
	 */
	public void setGameMode(GameModeEnum gameMode) {
		this.gameMode = gameMode;
	}

	public String getNazevMapy() {
		return nazevMapy;
	}

	public void setNazevMapy(String nazevMapy) {
		this.nazevMapy = nazevMapy;
	}

	public int getRychlostHry() {
		return rychlostHry;
	}

	protected void setRychlostHry(int rychlostHry) {
		this.rychlostHry = rychlostHry;
	}

	/**
	 * Pošle zprávu a sám jí nezobrazí. Pouze smaže textField pro zadání zprávy.
	 *
	 * @param zprava Zpráva na poslání.
	 */
	public void posliZpravu(String zprava) {
		net.poslatData(Informace.ZPRAVA + Informace.ODDELOVAC1 + zprava);
		setChanged();
		LobbyEvent evt = LobbyEvent.VYMAZ;
		evt.data = zprava;
		notifyObservers(evt);
	}

	/**
	 * Pošle zprávu a zobrazí ji v chatArea. Také smaže textField pro zadání zprávy.
	 *
	 * @param zprava Zpráva na poslání.
	 */
	public void zobrazAPosliZpravu(String zprava) {
		net.poslatData(Informace.ZPRAVA + Informace.ODDELOVAC1 + zprava);
		setChanged();
		LobbyEvent evt = LobbyEvent.ZPRAVA_A_VYMAZ;
		evt.data = zprava;
		notifyObservers(evt);
	}

	/**
	 * Zobrazí zprávu v chatArea.
	 *
	 * @param zprava Zpráva na poslání.
	 */
	public void zobrazZpravu(String zprava) {
		setChanged();
		LobbyEvent evt = LobbyEvent.ZPRAVA;
		evt.data = zprava;
		notifyObservers(evt);
	}
}
