package org.simplextensions.ui.menu;

import javassist.bytecode.annotation.Annotation;
import javassist.bytecode.annotation.StringMemberValue;
import org.hamcrest.Matchers;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mockito;
import org.mockito.runners.MockitoJUnitRunner;
import org.simplextensions.registry.Extension;
import org.simplextensions.registry.IExtensionRegistry;
import org.simplextensions.registry.SimpleXtensionsBundle;
import org.simplextensions.ui.menu.annotations.Action;
import org.simplextensions.ui.menu.annotations.Menu;
import org.simplextensions.ui.menu.scanner.MenuScannPhaseListener;

import static org.mockito.Matchers.argThat;
import static org.mockito.Matchers.eq;
import static org.mockito.Matchers.same;
import static org.mockito.Mockito.*;

@RunWith(MockitoJUnitRunner.class)
public class MenuScannPhaseListenerTest {

	private static final String SOME_CLASS_NAME = "SomeClassName";

	@Before
	public void setUp() {
		System.out.println("---------------------------------------------------------------------");
	}

	@After
	public void tearDown() {

	}

	@Test(expected = NullPointerException.class)
	public void testExceptionWhenNullArgument() {

		IExtensionRegistry mock = mock(IExtensionRegistry.class);

		MenuScannPhaseListener listener = mock(MenuScannPhaseListener.class);
		SimpleXtensionsBundle bundle = new SimpleXtensionsBundle(mock, "test");
		doCallRealMethod().when(listener).scanClass(same(mock), same(bundle), Mockito.any(String.class), Mockito.any(Annotation.class));

		listener.scanClass(mock, bundle, null, null);
	}

	@Test
	public void testAction() {

		IExtensionRegistry mock = mock(IExtensionRegistry.class);
		SimpleXtensionsBundle bundle = new SimpleXtensionsBundle(mock, "test");

		MenuScannPhaseListener listener = mock(MenuScannPhaseListener.class);
		doCallRealMethod().when(listener).scanClass(same(mock), same(bundle), eq(SOME_CLASS_NAME), Mockito.any(Annotation.class));

		Annotation annotation = mockAnnotation(Action.class);

		doThrow(new IllegalArgumentException("nieprawidlowa klasa")).when(mock).addExtension(
				(Extension) Mockito.argThat(Matchers.hasProperty("className", Matchers.not(Matchers.equalTo(SOME_CLASS_NAME)))), Mockito.anyString());

		listener.scanClass(mock, bundle, SOME_CLASS_NAME, annotation);
		verify(mock).addExtension(Mockito.any(Extension.class), Mockito.anyString());
	}

	@Test
	public void testMenu() {
		String parentIdValue = "value";
		String parentIdPropertyName = "parent";

		String idValue = "value";
		String idPropertyName = "id";

		IExtensionRegistry mock = mock(IExtensionRegistry.class);
		SimpleXtensionsBundle bundle = new SimpleXtensionsBundle(mock, "test");

		MenuScannPhaseListener listener = mock(MenuScannPhaseListener.class);
		doCallRealMethod().when(listener).scanClass(same(mock), same(bundle), eq(SOME_CLASS_NAME), Mockito.any(Annotation.class));

		StringMemberValue value = mock(StringMemberValue.class);
		when(value.getValue()).thenReturn(parentIdValue);
		StringMemberValue id = mock(StringMemberValue.class);
		when(id.getValue()).thenReturn(idValue);

		Annotation annotation = mockAnnotation(Menu.class);
		when(annotation.getMemberValue(parentIdPropertyName)).thenReturn(value);
		when(annotation.getMemberValue(idPropertyName)).thenReturn(id);

		doThrow(new IllegalArgumentException("nieporpawna wartosc properties")).when(mock).addExtension(
				argThat(Matchers.not(new ExtensionPropertyMatcher(parentIdPropertyName, parentIdValue))), Mockito.anyString());

		doThrow(new IllegalArgumentException("nieporawna klasa ")).when(mock).addExtension(
				(Extension) Mockito.argThat(Matchers.hasProperty("className", Matchers.not(Matchers.equalTo(SOME_CLASS_NAME)))), Mockito.anyString());

		doThrow(new IllegalArgumentException("nieporawne id")).when(mock).addExtension(
				(Extension) Mockito.argThat(Matchers.hasProperty("id", Matchers.not(Matchers.equalTo(idValue)))), Mockito.anyString());

		listener.scanClass(mock, bundle, SOME_CLASS_NAME, annotation);
		verify(mock).addExtension(Mockito.any(Extension.class), Mockito.anyString());

	}

	private Annotation mockAnnotation(Class<?> class1) {
		Annotation mock = mock(Annotation.class);

		when(mock.getTypeName()).thenReturn(class1.getCanonicalName());

		return mock;
	}

}
