package org.simplextensions.registry;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * 
 * @author Tomasz Krzyak, <a
 *         href="mailto:tomasz.krzyzak@gmail.com">tomasz.krzyzak@gmail.com</a>
 * @since 2009-07-29 21:58:02
 * 
 */
@org.simplextensions.annotations.ExtensionPoint(id = IService.EP_ID, extensionClass = IService.class)
public class ServiceRegistry implements IServiceRegistry {

	private static final Log log = LogFactory.getLog(ServiceRegistry.class);

	private Map<Class<?>, IService> servicesMap = new HashMap<Class<?>, IService>();
	private Map<Class<?>, Extension> extensionsMap = new HashMap<Class<?>, Extension>();

	public ServiceRegistry() {
		log.info("ServiceRegistry created...");
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.simplextensions.core.IIServiceRegistry#getService(java.lang.Class)
	 */
	@SuppressWarnings("unchecked")
	public <T> T getService(Class<T> serviceInterface) {
		if (!isStarted())
			return null;

		IService service = servicesMap.get(serviceInterface);
		if (service == null) {
			Extension extension = extensionsMap.get(serviceInterface);
			if (extension != null) {
				service = (IService) extension.getExecutable();
				log.info("Initializing service: " + serviceInterface.getCanonicalName());
				service.init(this);
				servicesMap.put((Class<? extends IService>) serviceInterface, service);
			}
		}

		if (!service.isStarted()) {
			log.info("Starting service: " + serviceInterface.getCanonicalName());
			service.start();
		}
		return (T) service;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.simplextensions.core.IIServiceRegistry#registerService(java.lang.
	 * Class, K)
	 */
	public synchronized <T, K extends IService> void registerService(Class<T> serviceInterface, K service) {
		if (log.isInfoEnabled())
			log.info("Registering service: " + service.getClass().getCanonicalName() + " with interface: "
					+ serviceInterface.getCanonicalName());

		service.init(this);
		
		this.servicesMap.put(serviceInterface, service);
		if (started) {
			service.start();
		}
	}

	boolean started = false;

	public boolean isStarted() {
		return started;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.simplextensions.registry.IServiceRegistry#start()
	 */
	public synchronized void start() {
		if (started) {
			throw new RuntimeException("already started");
		}
		log.info("Starting Service Registry");
		for (IService service : servicesMap.values()) {
			try {
				service.start();
			} catch (Exception e) {
				log.error("", e);
			}
		}
		started = true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.simplextensions.registry.IServiceRegistry#stop()
	 */
	public void stop() {
		if (!started) {
			throw new RuntimeException("not yet started");
		}
		log.info("Stopping Service Registry");
		for (IService service : servicesMap.values()) {
			try {
				service.stop();
			} catch (Exception e) {
				log.error("", e);
			}
		}
	}

	@SuppressWarnings("unchecked")
	public void initialize(ExtensionPoint extensionPoint) {
		Collection<Extension> extensions = extensionPoint.getActiveExtensions();
		for (Extension e : extensions) {
			PropertyValue propertyValue = e.getPropertyValue(IService.SERVICE_INTERFACE);
			if (propertyValue == null) {
				log.warn(e.getId() + " does not define parameter: " + IService.SERVICE_INTERFACE);
				continue;
			}
			Class<?> classValue = propertyValue.getClassValue();
			if (classValue != null && IService.class.isAssignableFrom(e.getClazz())) {
				extensionsMap.put((Class<? extends IService>) classValue, e);
				log.debug("registering service: " + e.getId() + " binded to iface: " + classValue.getCanonicalName());
			}
		}
	}

}
