/**
 * 
 */
package org.simplextensions.scanner;

import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.Enumeration;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

import javassist.bytecode.AnnotationsAttribute;
import javassist.bytecode.ClassFile;
import javassist.bytecode.annotation.Annotation;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.simplextensions.registry.Bundle;

/**
 * @author Tomasz Krzyak, <a
 *         href="mailto:tomasz.krzyzak@gmail.com">tomasz.krzyzak@gmail.com</a>
 * @since 2009-08-18 22:30:07
 * 
 */
public class ClassPathScanner {

	private static final Log log = LogFactory.getLog(ClassPathScanner.class);

	public ClassPathScanner() {
	}

	protected boolean isJARPath(String path) {
		File file = new File(path);
		return file.isFile() && file.getName().endsWith(".jar");
	}

	private long jarTime = 0;
	private long classFileTime = 0;
	private long listenersTime = 0;

	protected void scanJARFile(Bundle bundle, String jarPath) throws IOException {
		log.info("Scanning jar file: " + jarPath);
		JarFile jarFile = new JarFile(jarPath);

		Set<String> classNames = bundle.getClassNames();
		Enumeration<JarEntry> entries = jarFile.entries();
		JarEntry nextJarEntry = null;
		try {
			while (entries.hasMoreElements()) {
				nextJarEntry = entries.nextElement();
				String name = nextJarEntry.getName();
				
				//log.error("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@"+name);
				
				boolean scannEntry = false;

				if (classNames != null) {
					if (classNames.contains(name.substring(0, name.length() - (name.length() > 6 ? 6 : 0)).replace("/", "."))) {
						scannEntry = true;
					}
				} else {
					if (name.endsWith(".class")) {
						scannEntry = true;
					}
				}

				if (scannEntry) {
					InputStream inputStream = null;
					try {
						inputStream = jarFile.getInputStream(nextJarEntry);
						ClassFile classFile = new ClassFile(new DataInputStream(inputStream));
						scanClassFile(bundle, classFile);
					} catch (Exception e) {
						throw new ScannerException(name, e);
					} finally {
						if (inputStream != null)
							inputStream.close();
					}
				}
			}
		} finally {
			jarFile.close();
		}
	}

	/**
	 * @param directory
	 * @param string
	 * @return
	 * @throws IOException
	 */
	private void scanDirectory(Bundle bundle, File directory, String basePath) throws IOException {
		Set<String> classNames = bundle.getClassNames();

		if (directory.exists()) {
			for (File file : directory.listFiles()) {
				if (file.isFile()) {
					if (file.getName().endsWith(".class")) {
						boolean scannEntry = false;
						if (classNames != null) {
							String classFilePath = file.getCanonicalPath().substring(basePath.length() + 1);
							classFilePath = classFilePath.substring(0, classFilePath.length() - (classFilePath.length() > 6 ? 6 : 0)).replace(File.separatorChar, '.');

							if (classNames.contains(classFilePath)) {
								scannEntry = true;
							}
						} else {
							scannEntry = true;
						}
						if (scannEntry) {
							FileInputStream fileInputStream = null;
							try {

								fileInputStream = new FileInputStream(file);
								ClassFile classFile = new ClassFile(new DataInputStream(fileInputStream));
								scanClassFile(bundle, classFile);

							} catch (Exception e) {
								throw new ScannerException("", e);
							} finally {
								if (fileInputStream != null)
									fileInputStream.close();
							}
						}

					}
				} else
					scanDirectory(bundle, file, basePath);
			}
		}
	}

	public void scannClassPath(Collection<Bundle> bundles) {

		jarTime = -System.nanoTime();
		for (Bundle b : bundles) {
			String path = b.getClassesPath();
			try {
				long a = System.nanoTime();
				if (isJARPath(path)) {
					scanJARFile(b, path);
				} else {
					scanDirectory(b, new File(path), new File(path).getCanonicalPath());
				}
				log.info("Scanning time:" + (System.nanoTime() - a) / 1000000.0 + "ms of Bundle: " + b.getName());
			} catch (Exception e) {
				throw new ScannerException(path, e);
			}
		}
		jarTime += System.nanoTime();

		log.info("JarFile scanning time: " + jarTime / 1000000.0 + " (classFileTime: " + classFileTime / 1000000.0 + "ms(listenersTime: "
				+ listenersTime / 1000000.0 + "ms))");
	}

	/**
	 * @param classFile
	 */
	private void scanClassFile(Bundle bundle, ClassFile classFile) {
		classFileTime -= System.nanoTime();
		AnnotationsAttribute attributeInfo = (AnnotationsAttribute) classFile.getAttribute(AnnotationsAttribute.invisibleTag);
		if (attributeInfo != null)
			scanClassFile(bundle, classFile, attributeInfo);
		else {
			attributeInfo = (AnnotationsAttribute) classFile.getAttribute(AnnotationsAttribute.visibleTag);
			scanClassFile(bundle, classFile, attributeInfo);
		}
		classFileTime += System.nanoTime();
	}

	/**
	 * @param classFile
	 * @param attributeInfo
	 */
	private void scanClassFile(Bundle bundle, ClassFile classFile, AnnotationsAttribute attributeInfo) {
		if (attributeInfo != null) {
			Map<String, Annotation> annotationsMap = new LinkedHashMap<String, Annotation>();
			for (Annotation a : attributeInfo.getAnnotations()) {
				annotationsMap.put(a.getTypeName(), a);
			}
			notifyScanListeners(bundle, classFile, annotationsMap);
		}
	}

	private void notifyScanListeners(Bundle bundle, ClassFile classFile, Map<String, Annotation> annotationsMap) {
		listenersTime -= System.nanoTime();
		for (Annotation a : annotationsMap.values()) {
			List<IScanPhaseListener> scanListeners = bundle.getScanListeners(a.getTypeName());
			if (scanListeners != null)
				for (IScanPhaseListener listener : scanListeners) {
					listener.scanClass(classFile, a);
				}
			else {
				log.info("Unsupported annotation: " + a.getTypeName());
			}
		}
		listenersTime += System.nanoTime();
	}
}
