package org.simplextensions.ui.preferences;

import org.simplextensions.annotations.Service;
import org.simplextensions.di.annotations.ServiceDependency;
import org.simplextensions.registry.AService;
import org.simplextensions.registry.IServiceRegistry;
import org.simplextensions.ui.ISimpleXtensionsUIService;
import org.simplextensions.ui.preferences.xml.XMLPreferencesPersister;

/**
 * 
 * This implementation of {@link IPreferencesService} uses
 * {@link XMLPreferencesPersister} as default properties persister. <br />
 * <br />
 * If you woulde like to change default persistence mechanism you have to
 * <ul>
 * <li>implement your own {@link IPreferencesPersister} and {@link IPreferences}
 * <li>create <b>simplextensions-ui.properties</b> file and inside this file set
 * property <b>preferencesservice.persister.className</b> or
 * <b>preferencesservice.persister.factoryClassName</b> with a value of your own
 * persister class name (if both of these properties are specified
 * <b>preferencesservice.persister.className</b> is prefered)
 * </ul>
 * 
 * 
 * @author krzyzak
 * 
 */
@Service(IPreferencesService.class)
public class PreferencesService extends AService implements IPreferencesService {

	public static final String PersisterPropertyName = "preferencesservice.persister.className";
	public static final String PersisterFactoryPropertyName = "preferencesservice.persister.factoryClassName";

	private IPreferencesPersister persister;

	@ServiceDependency
	private ISimpleXtensionsUIService sxUIService;

	public void start() {
		String persisterClassName = getSxUIService().getProperty(PersisterPropertyName);
		String persisterFactoryClassName = getSxUIService().getProperty(PersisterPropertyName);

		if (persisterClassName != null) {
			try {
				persister = (IPreferencesPersister) Class.forName(persisterClassName).newInstance();
			} catch (Exception e) {
				throw new RuntimeException(e);
			}
		} else if (persisterFactoryClassName != null) {
			try {
				IPreferencesPersisterFactory factory = (IPreferencesPersisterFactory) Class.forName(persisterClassName).newInstance();
				persister = factory.createPersister(getServiceRegistry());
			} catch (Exception e) {
				throw new RuntimeException(e);
			}
		}

		if (persister == null) {
			persister = new XMLPreferencesPersister();
		}

		persister.init(getServiceRegistry());

		super.start();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.simplextensions.ui.preferences.IPreferencesService#get(java.lang.
	 * String, java.lang.String)
	 */
	public IPreferences get(String id, String bundleId) {
		return this.persister.get(id, bundleId);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @seeorg.simplextensions.ui.preferences.IPreferencesService#save(org.
	 * simplextensions.ui.preferences.IPreferences)
	 */
	public void save(IPreferences preferences) {
		this.persister.save(preferences);
	}

	public ISimpleXtensionsUIService getSxUIService() {
		return sxUIService;
	}

}
