package org.simplextensions.ui.menu;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MenuAdapter;
import org.eclipse.swt.events.MenuEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.simplextensions.annotations.ioc.Point;
import org.simplextensions.ioc.DependencyInjector;
import org.simplextensions.annotations.ioc.Registry;
import org.simplextensions.registry.IExtensionRegistry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 
 * 
 * @author Tomasz Krzy�ak, <a
 *         href="mailto:tomasz.krzyzak@gmail.com">tomasz.krzyzak@gmail.com</a>
 * @since 2010-05-17 10:11:52
 */
public abstract class MenuManager {

	private static final Logger log = LoggerFactory.getLogger(MenuManager.class);

	@Registry
	private IExtensionRegistry extensionRegistry;

	@Point
	private MenuExtensionPoint mep;

	protected IExtensionRegistry getExtensionRegistry() {
		return extensionRegistry;
	}

	protected MenuExtensionPoint getMep() {
		return mep;
	}

	public MenuManager(IExtensionRegistry extensionRegistry) {
		DependencyInjector.resolveDependency(extensionRegistry, this);
	}

	protected void createSubMenu(Menu parentMenu, MenuExtensionMetadata memd) {
		if (memd.getType() == MenuExtensionType.Menu) {
			try {
				MenuItem subMenuItem = new MenuItem(parentMenu, SWT.CASCADE);
				subMenuItem.setText(memd.getExecutable().getLabel(memd));

				Menu subMenu = new Menu(subMenuItem);
				subMenuItem.setMenu(subMenu);

				for (MenuExtensionMetadata submemd : memd.getSubItems()) {
					if (submemd.getType() == MenuExtensionType.Menu) {
						createSubMenu(subMenu, submemd);
					} else if (submemd.getType() == MenuExtensionType.Separator) {
						new MenuItem(subMenu, SWT.SEPARATOR);
					} else if (submemd.getType() == MenuExtensionType.Action) {
						addAction(subMenu, submemd);
					}
				}
			} catch (Exception e) {
				log.error("", e);
			}
		} else {
			log.warn(memd.getType().name() + " not supported in MenuBar (" + memd.toString() + ")");
		}
	}

	protected void addAction(Menu parentMenu, MenuExtensionMetadata submemd) {
		if (submemd.getType() == MenuExtensionType.Action) {

			IMenuExtension executable = submemd.getExecutable();
			if (executable instanceof IActions) {
				createMenuItems(parentMenu, submemd, (IActions) executable);
			} else if (executable instanceof IAction) {
				createMenuItem(parentMenu, submemd, (IAction) executable);
			}
		}
	}

	protected void createMenuItems(Menu parentMenu, final MenuExtensionMetadata submemd, final IActions executable) {
		MenuItem menuItem = new MenuItem(parentMenu, SWT.CASCADE);
		menuItem.setText(executable.getLabel(submemd));

		final Menu subMenu = new Menu(menuItem);
		menuItem.setMenu(subMenu);

		subMenu.addMenuListener(new MenuAdapter() {
			public void menuShown(MenuEvent arg0) {
				fillMenu(submemd, executable.getActions(), subMenu);
			}
		});
	}

	protected void createMenuItem(Menu parentMenu, final MenuExtensionMetadata submemd, final IAction executable) {
		MenuItem pushMenuItem = new MenuItem(parentMenu, SWT.PUSH);
		pushMenuItem.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent arg0) {
				executable.run(submemd);
			}
		});
		pushMenuItem.setText(executable.getLabel(submemd));
		pushMenuItem.setEnabled(executable.isEnabled(submemd));
	}

	protected void fillMenu(MenuExtensionMetadata submemd, IAction[] actions, Menu parentMenu) {
		for (MenuItem mi : parentMenu.getItems()) {
			mi.dispose();
		}
		for (IAction action : actions) {
			createMenuItem(parentMenu, submemd, action);
		}
	}

}
