/*
 * Copyright (c) 2004-2008, Volatile-Engine All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * Neither the name of the Volatile-Engine nor the
 * names of its contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */
package com.volatileengine.xml.elements;

import java.util.HashMap;

import com.volatileengine.xml.IFileLoader;

/**
 * 
 * <code>Element</code> represents an object that is defined in a file.
 * <code>Element</code> has the responsibility of creating and validating child
 * <code>Element</code>s in order to validate the incoming file as well as
 * process it.
 * 
 * See methods below for implementation details
 * 
 * 
 * @author darkprophet
 * 
 * @param <T>
 */
public interface Element<T> {

	/**
	 * Get the value of this Element in the file.
	 * 
	 * @return
	 */
	public T getValue();

	/**
	 * Add more data to this element as it comes across.
	 * 
	 * @param addtext
	 */
	public void append(byte[] addtext);

	/**
	 * Return the attributes associated with this Element, but not the data
	 * itself.
	 * 
	 * @return
	 */
	public HashMap<String, String> getAttributes();

	/**
	 * Get the associated tag/block of the Element in the data file
	 * 
	 * @return
	 */
	public abstract String getAssociatedTag();

	/**
	 * This method is called when a new tag has been identified in the data
	 * file. Instantiate a new class accordingly or through exceptions if the
	 * child is not approperiate.
	 * 
	 * @param tagName
	 * @param attributes
	 * @param loader
	 * @return
	 */
	public Element<?> instantiateChild(String tagName, IFileLoader<?> loader);

	/**
	 * This method is called when the element start is found.
	 * 
	 * @param loader
	 */
	public abstract void startElement(HashMap<String, String> attributes, IFileLoader<?> loader);

	/**
	 * This method is called when the child of this Element has ended in the
	 * data. It is now the responsibility of the class to process the children.
	 * 
	 * @param child
	 * @param loader
	 */
	public abstract void processChild(Element<?> child, IFileLoader<?> loader);

	/**
	 * This method is called when the end tag of this Element has been found.
	 * This is the end of this Element in the data file
	 * 
	 * @param loader
	 */
	public abstract void endElement(IFileLoader<?> loader);
}
