/*    
 * Copyright (C) 2006  ankostis, Stefan Kuper

 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

package net.googlecode.weblogic_jaxb1codec;

import java.io.IOException;
import java.io.Reader;

import weblogic.xml.stream.Attribute;
import weblogic.xml.stream.AttributeIterator;
import weblogic.xml.stream.CharacterData;
import weblogic.xml.stream.Space;
import weblogic.xml.stream.XMLEvent;
import weblogic.xml.stream.XMLInputStream;
import weblogic.xml.stream.XMLStreamException;
import weblogic.xml.stream.events.StartElementEvent;

/**
 * @author SK, ankostis
 */
class DeserializerReader extends Reader {

    private int nodeDepth;

	private XMLInputStream inputStream;

	private StringBuffer buffer = new StringBuffer();

//    private byte[] byteBuffer = null;

    private int marker;

    /**
     * Usefull for closing empty elements directly,
     * that is, to write '<someElement/>' instead of '<someElement></someElement>.
     */
    private boolean elementStarted = false;

	private boolean finishedReading = false;

	/**
	 * @param rootName
	 *            The top XML element name.
	 * @param inputStream
	 *            The {@link XMLInputStream}.
	 * @throws XMLStreamException 
	 */
	public DeserializerReader(XMLInputStream inputStream) {
		this.inputStream = inputStream;
	}

    public void close() throws IOException {
        this.inputStream.close();
    }

    public int read(char[] outbuf, int destOffset, int maxChars) throws IOException {
        if (finishedReading) {
            return -1;
        }
    
        while (!finishedReading && buffer.length() == 0) {
            buildXMLEventString();
        }
        
        int charsLeft = buffer.length() - marker;
        int charsToCopy = Math.min(charsLeft, maxChars);
        int lastChar = marker + charsToCopy;
        
        buffer.getChars(marker, lastChar, outbuf, destOffset);
        
        if (charsLeft == charsToCopy) {
            buffer.delete(0, buffer.length());
            marker = 0;
        } else
            marker = lastChar;

        return charsToCopy;
    }

	/**
	 * @throws XMLStreamException
	 * 
	 */
	private void buildXMLEventString() throws XMLStreamException {
		if (inputStream.hasNext()) {
			XMLEvent ev = inputStream.next();

            // The next piece of code is for
            // closing empty elements directly,
            // ie to write '<someElement/>' instead of '<someElement></someElement>
            if (ev.isEndElement()) {
                if (elementStarted)
                    buffer.append("/>");
                else
                    buffer.append("</" + ev.getName().getQualifiedName() + ">");

                // Update node depth.
                nodeDepth--;
                
                // Skip reading elements after our top-most element.
                if (nodeDepth == 0) {
                    finishedReading = true;
                    return;
                } else if (nodeDepth <= 0) {
                    throw new XMLStreamException("Unexpected situation while deserializing input, read to many nodes! Current depth: " + nodeDepth);
                }
                
            } else if (elementStarted) {
                buffer.append(">");
            }

            elementStarted = false;

            
            switch (ev.getType()) {
            case XMLEvent.START_ELEMENT:
                elementStarted = true;

                StartElementEvent el = (StartElementEvent) ev;

                buffer.append("<" + el.getName().getQualifiedName());

                AttributeIterator it = el.getAttributesAndNamespaces();
                while (it.hasNext()) {
                    Attribute at = it.next();
                    buffer.append(" " + at.getName().getQualifiedName() + "='" + at.getValue() + "'");
                }

                // Update node depth.
                nodeDepth++;

                break;
            case XMLEvent.CHARACTER_DATA:
                CharacterData cdata = (CharacterData) ev;
                if (cdata.hasContent())
                    buffer.append(cdata.getContent());
                if (cdata.getContent().startsWith(""))
                    System.out.println("\n##############: "+cdata.getContent());
                
                break;
            case XMLEvent.SPACE:
                Space spc = (Space) ev;
                if (spc.hasContent())
                    buffer.append(spc.getContent());
                
                break;
            case XMLEvent.END_DOCUMENT:
            case XMLEvent.END_ELEMENT:
            case XMLEvent.END_PREFIX_MAPPING:
            case XMLEvent.START_PREFIX_MAPPING:
            case XMLEvent.START_DOCUMENT:
//                System.out.println("\nDeserl event["+ev.getTypeAsString() +"]: " + ev);
                
                break;
            default:
                System.out.println("\nUNKNOWN Deserl event["+ev.getTypeAsString() +"]: " + ev);
                break;
            }

		} else {
			finishedReading = true;
		}
	}

}
