import java.util.*;
import java.io.*;
import java.text.NumberFormat;

class AffinityPropagation extends InfiniteExemplarModel {

    /*****************************************************************/
    /*                     Instance Variables                        */
    /*****************************************************************/

    // Internal messages that are currently on the edges between nodes in
    // the factor graph.
    double availabilities_[][];
    double responsibilities_[][];

    // Allow messages to be computed and stored here without automatically
    // applying the updates. Separating this computation from updates allows 
    // for more flexible message scheduling.
    double newAvailabilities_[][];
    double newResponsibilities_[][];

    int[] tunedAssignment_;

    // For logging and monitoring inference
    int iteration_;
    int messagesComputed_;
    int messagesUpdated_;

    // Tunable parameters
    DPAPParameters parameters_;
    double dampingFactor_;

    // Nice printing
    NumberFormat twoDigitNF_;

    /*****************************************************************/
    /*                            Methods                            */
    /*****************************************************************/

    /**
     * Simple default constructor.
     */
    public AffinityPropagation(int numVars) {
	numVars_ = numVars;
	dampingFactor_ = .7;
	iteration_ = 0;

	similarities_ = new double[numVars_][numVars_];
	alpha_ = 1;

	messagesComputed_ = 0;
	messagesUpdated_ = 0;

	tunedAssignment_ = null;

	twoDigitNF_ = NumberFormat.getInstance();
	twoDigitNF_.setMaximumFractionDigits(2);

	InitializeMessages();
    }

    /**
     * Construct from parameter file.
     */
    public AffinityPropagation(String parameterFile) {
	assert false : "Using bad AP constructor";
    }

    // Getters and setters
    public int Iteration() {
	return iteration_;
    }
    
    public void PrintStats() {
	System.out.println("Computed " + messagesComputed_ + " messages");
	System.out.println("Updated " + messagesUpdated_ + " messages");
    }

    /**
     * Keep track of the name of the algorithm being implemented.
     */
    public String FullName() { return "Affinity Propagation"; }
    public String ShortName() { return "AP"; }

    /**
     * Copy parameters from the parameter file into the actual instance
     * variables that are relevant.
     */
    protected void SyncParameters(Parameters p) {
	System.out.println("Parameter file: " + parameters_.parameterFile_);
	System.out.println("Syncing AP Params: " + parameters_.FileName());
	dampingFactor_ = parameters_.dampingFactor_;
    }

    /**
     * Zero out availabilities and responsibilities.
     */
    protected void InitializeMessages() {
	availabilities_ = new double[numVars_][numVars_];
	newAvailabilities_ = new double[numVars_][numVars_];
	responsibilities_ = new double[numVars_][numVars_];
	newResponsibilities_ = new double[numVars_][numVars_];

	for (int i = 0; i < numVars_; i++) {
	    for (int j = 0; j < numVars_; j++) {
		availabilities_[i][j] = 0;
		newAvailabilities_[i][j] = -Double.MAX_VALUE;
		responsibilities_[i][j] = 0;
		newResponsibilities_[i][j] = -Double.MAX_VALUE;
	    }
	}
    }

    /**
     * Given the current messages on the edges, compute the beliefs for
     * each variable and use it to choose exemplars.
     */
    public ArrayList<Integer> CurrentExemplars() {
	ArrayList<Integer> exemplars = new ArrayList<Integer>();
	for (int j = 0; j < numVars_; j++) {
	    if (availabilities_[j][j] + responsibilities_[j][j] > 0.0) {
		exemplars.add(new Integer(j));
	    }
	}
	return exemplars;
    }

    /**
     * Choose exemplars, then just assign each point to the nearest exemplar.
     */
    public int[] CurrentAssignments() {
	ArrayList<Integer> exemplars = CurrentExemplars();
	int[] assignment = new int[numVars_];

	for (int i = 0; i < numVars_; i++) {
	    assignment[i] = -1;
	    double maxBelief = -Double.MAX_VALUE;
	    for (int j = 0; j < exemplars.size(); j++) {
		if (similarities_[i][exemplars.get(j)] > maxBelief) {
		    maxBelief = similarities_[i][exemplars.get(j)];
		    assignment[i] = exemplars.get(j);
		}
	    }
	}

	return assignment;
    }

    /**
     * Compute, but don't apply update to responsibility of point
     * to exemplar.  Separating computation from updates allows for
     * more flexible message scheduling.
     */
    protected void ComputeResponsibility(int point, int exemplar) {
	double firstMax = -Double.MAX_VALUE;
	double secondMax = -Double.MAX_VALUE;
	int maxIndex = -1;
	for (int j = 0; j < numVars_; j++) {
	    double AS = similarities_[point][j] + availabilities_[point][j];
	    if (AS > firstMax) {
		secondMax = firstMax;
		firstMax = AS;
		maxIndex = j;
	    } else if (AS > secondMax) {
		secondMax = AS;
	    }
	}

	double rIJ;
	if (maxIndex == exemplar) {
	    rIJ = similarities_[point][exemplar] - secondMax;	
	} else {
	    rIJ = similarities_[point][exemplar] - firstMax;	
	}

	newResponsibilities_[point][exemplar] = rIJ;
	messagesComputed_++;
    }

    /**
     * Apply the update, assuming that it has already been computed.
     */
    protected void UpdateResponsibility(int point, int exemplar) {
	// Make sure message has been computed before updating
	assert(newResponsibilities_[point][exemplar] != -Double.MAX_VALUE);
	
	responsibilities_[point][exemplar] = 
	    dampingFactor_ * responsibilities_[point][exemplar] + 
	    (1 - dampingFactor_) * newResponsibilities_[point][exemplar];
	
	newResponsibilities_[point][exemplar] = -Double.MAX_VALUE;
	messagesUpdated_++;
    }

    /**
     * Compute, but don't apply update to responsibility of point
     * to exemplar.  Separating computation from updates allows for
     * more flexible message scheduling.
     */
    protected void ComputeAvailability(int point, int exemplar) {
	double sumResponsibilities = 0;
	for (int j = 0; j < numVars_; j++) {
	    if (j != point && j != exemplar) {
		sumResponsibilities += 
		    Math.max(responsibilities_[point][exemplar], 0.0);
	    }
	}

	double aIJ;
	if (point == exemplar) {
	    aIJ = sumResponsibilities;
	} else {
	    aIJ = Math.min(0.0, responsibilities_[exemplar][exemplar] +
			   sumResponsibilities);
	}

	newAvailabilities_[point][exemplar] = aIJ;
	messagesComputed_++;
    }

    /**
     * Apply the update, assuming that it has already been computed.
     */
    protected void UpdateAvailability(int point, int exemplar) {
	// Make sure message has been computed before updating
	assert(newAvailabilities_[point][exemplar] != -Double.MAX_VALUE);
	
	availabilities_[point][exemplar] = 
	    dampingFactor_ * availabilities_[point][exemplar] + 
	    (1 - dampingFactor_) * newAvailabilities_[point][exemplar];

	newAvailabilities_[point][exemplar] = -Double.MAX_VALUE;
	messagesUpdated_++;
    }
    
    /**
     * Decide which messages to update.  Default is an alternating 
     * synchronous schedule.
     */
    protected ArrayList<ScheduledMessage> MessagesToUpdate() {
	ArrayList<ScheduledMessage> messages = 
	    new ArrayList<ScheduledMessage>();
	for (int i = 0; i < numVars_; i++) {
	    for (int j = 0; j < numVars_; j++) {
		int messageType;
		if (iteration_ % 2 == 0) {
		    messageType = ScheduledMessage.AVAILABILITY;
		} else {
		    messageType = ScheduledMessage.RESPONSIBILITY;
		}
		ScheduledMessage message = 
		    new ScheduledMessage(messageType, i, j);
		messages.add(message);
	    }
	}

	return messages;
    }

    /**
     * Do one round of message passing.
     */
    public void PassMessages() {
	ArrayList<ScheduledMessage> messages = MessagesToUpdate();
	
	// Compute the messages
	for (int i = 0; i < messages.size(); i++) {
	    if (messages.get(i).Type() == ScheduledMessage.AVAILABILITY) {
		ComputeAvailability(messages.get(i).Point(),
				    messages.get(i).Exemplar());
	    } else if (messages.get(i).Type() == 
		       ScheduledMessage.RESPONSIBILITY) {
		ComputeResponsibility(messages.get(i).Point(),
				      messages.get(i).Exemplar());		
	    }
	}

	// Actually apply the updates
	for (int i = 0; i < messages.size(); i++) {
	    if (messages.get(i).Type() == ScheduledMessage.AVAILABILITY) {
		UpdateAvailability(messages.get(i).Point(),
				   messages.get(i).Exemplar());
	    } else if (messages.get(i).Type() == 
		       ScheduledMessage.RESPONSIBILITY) {
		UpdateResponsibility(messages.get(i).Point(),
				     messages.get(i).Exemplar());		
	    }
	}
	
	iteration_++;
    }

    /**
     *  Do the inference.
     */
    public void MAPInference() {
	while (!HasConverged()) {
	    PassMessages();
	    //System.err.print(Iteration() + " ");

	    /*
	    ArrayList<Integer> exemplars = CurrentExemplars();
	    System.out.println("Exemplars: " + exemplars);
	    
	    int[] assignments = CurrentAssignments();
	    System.out.println("Full Assignment: ");
	    for (int i = 0; i < assignments.length; i++) {
		System.out.print(assignments[i] + " ");
	    }
	    System.out.println();
	    */
	}
    }

    /**
     * Largest norm of messages on the channel.
     */
    public boolean HasConverged() {
	return Iteration() > 500;
    }

    public void AddToDiagonal(double d) {
	for (int i = 0; i < numVars_; i++) {
	    similarities_[i][i] += d;
	}
    }

    /**
     * Given a filename, load parameters.  The file should be a text file
     * of the following form:
     * <parameter1_name> <parameter1_value>
     * ...
     * <parameterN_name> <parameterN_value>
     */
    public void LoadParametersFromFile(String filename) {
	try {
	    BufferedReader input = 
		new BufferedReader(new FileReader(filename));
	    
	    // There's just one line to read
	    String line ;
	    while ( (line = input.readLine()) != null ) {
		String[] entries = line.trim().split("\\s+");		
		
		// Allow blank lines
		if (entries.length == 0) continue;
		
		// Otherwise, there must be 2 entries per line
		assert(entries.length == 2);
		
		if (entries[0].equals("dampingFactor")) {
		    dampingFactor_ = Double.parseDouble(entries[1]);
		} else {
		    System.err.println("Warning: unknown parameter " +
				       entries[0] + ", value = " + entries[1]);
		}
	    }
	}
	catch (Exception ex){
	    ex.printStackTrace();
	    System.exit(1);
	}
    }
	
    /*****************************************************************/
    /*                              Main                             */
    /*****************************************************************/
    
    public static void main(String args[]) {
	AffinityPropagation ap = new AffinityPropagation(100);
	
	ap.LoadSimilaritiesFromFile(args[0]);

	if (args.length >= 2) {
	    ap.AddToDiagonal(Double.parseDouble(args[1]));
	}

	ap.MAPInference();

	ArrayList<Integer> exemplars = ap.CurrentExemplars();
	System.out.println("Exemplars: " + exemplars);
	
	int[] assignments = ap.CurrentAssignments();
	System.out.println("Full Assignment: ");
	for (int i = 0; i < assignments.length; i++) {
	    System.out.print(assignments[i] + " ");
	}
	System.out.println();
	
	ap.PrintStats();
    }

}
