package test;

import static base.Edge.DO_NOTHING;
import static base.Edge.NOT_TURN_ON;
import static base.Edge.TURN_OFF;
import static base.Edge.TURN_ON;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.junit.Test;

import base.Edge;
import base.SupportState;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;


public class PerformanceTests {

	private static final int cap = Integer.MAX_VALUE ;//>>> 1;
	
	private Set<Edge> testImmutable;
	private Set<Edge> testCUE;
	
	//@Before
	public void setUpEnumVsCUE() {
		testImmutable = Sets.immutableEnumSet(DO_NOTHING, TURN_OFF, TURN_ON);
		testCUE = Collections.unmodifiableSet(EnumSet.of(DO_NOTHING, TURN_OFF, TURN_ON));
	}
	
	//@Test
	public void timeImmutableEnumCreation() {
		Set<Edge> edges;
		boolean result = false;
		for (int i=0; i < cap; i++) {
			edges = Sets.immutableEnumSet(DO_NOTHING, TURN_OFF, TURN_ON);
			result |= edges.contains(NOT_TURN_ON);
		}
	}

	//@Test
	public void timeCollectionUnmodifiableEnumCreation() {
		Set<Edge> edges;
		boolean result = false;
		for (int i=0; i < cap; i++) {
			edges = Collections.unmodifiableSet(EnumSet.of(DO_NOTHING, TURN_OFF, TURN_ON));
			result |= edges.contains(NOT_TURN_ON);
		}
	}

	//@Test
	public void timeImmutableEnumIteration() {
		int result = 0;
		while (result < cap) for (Edge e : testImmutable) if (e!=DO_NOTHING) result++;
	}
	
	//@Test
	public void timeCollectionUnmodifiableEnumIteration() {
		int result = 0;
		while (result < cap) for (Edge e : testCUE) if (e!=DO_NOTHING) result++;
	}

	//@Test
	public void timeImmutableListIteration() {
		List<Edge> all = ImmutableList.copyOf(Edge.values());
		int result = Integer.MIN_VALUE;
		while (result < cap) for (Edge e : all) if (e!=DO_NOTHING) result++;
	}
	
	//@Test
	public void timeCUAListIteration() {
		List<Edge> all = Collections.unmodifiableList(Arrays.asList(Edge.values()));
		int result = Integer.MIN_VALUE;
		while (result < cap) for (Edge e : all) if (e!=DO_NOTHING) result++;
	}
	
	private Map<SupportState,SupportState> hash;
	private Map<SupportState,SupportState> conc;
	private static List<SupportState> ref;
	private List<Runnable> hashThreads;
	private List<Runnable> concThreads;
	
	public static class Reader implements Runnable {

		private final Map<SupportState,SupportState> test;
		private final Random rng = new Random();
		Object[] local = ref.toArray();

		
		Reader(Map<SupportState,SupportState> ref) {
			test = ref;
		}
		
		@Override public void run() {
			int index;
			boolean result = true;
			for (int i=0; i < cap; i++) {
				index = rng.nextInt(local.length);
				result = test.get(local[index]) == local[index] && result;
			}
		}
		
	}
	
	private ExecutorService e;
	
	//@Before
	public void setUpHashVsConcurrent() {
		int size = 7;
		hash = new HashMap<SupportState,SupportState>();
		conc = new ConcurrentHashMap<SupportState,SupportState>();
		ref = Lists.newArrayList(SupportState.allOfSize(size));
		for (SupportState s : ref) {
			hash.put(s, s);
			conc.put(s, s);
		}
		
		hashThreads = new ArrayList<Runnable>(size);
		concThreads = new ArrayList<Runnable>(size);

		for (int i=0; i<size*size ; i++) {
			hashThreads.add(new Reader(hash));
			concThreads.add(new Reader(conc));
		}
		
		e = Executors.newFixedThreadPool(size*size);
	}
	
	//@Test
	public void timeHashRead() {
		for (Runnable r : hashThreads) e.execute(r);
		e.shutdown();
	}
	
	//@Test
	public void timeConcurrentRead() {
		for (Runnable r : concThreads) e.execute(r);
		e.shutdown();
	}
	
}
