package rates;

import common.BusinessDayConvention;
import common.Date;
import common.DayCountConvention;
import common.Frequency;
import common.TimeUnit;

public class Bond {
	protected Date issueDate;
	protected Date maturityDate;
	protected Date firstCouponDate;
	protected double coupon;
	protected Frequency frequency;
	protected double faceAmount;
	protected YieldCurve yieldCurve;
	protected DayCountConvention dayCountConvention;
	
	public Bond(Date issueDate, Date maturityDate, Date firstCouponDate, 
			double coupon, Frequency frequency, double faceAmount, 
			DayCountConvention dayCountConvention, YieldCurve yieldCurve) {
		this.issueDate = new Date(issueDate);
		this.maturityDate = new Date(maturityDate);
		this.firstCouponDate = new Date(firstCouponDate);
		this.coupon = coupon;
		this.frequency = frequency;
		this.faceAmount = faceAmount;
		this.dayCountConvention = dayCountConvention;
		this.yieldCurve = new YieldCurve(yieldCurve);
	}
	
	public Bond(Bond bond) {
		this(bond.getIssueDate(), bond.getMaturityDate(), bond.getFirstCouponDate(),
				bond.getCoupon(), bond.getFrequency(), bond.getFaceAmount(),
				bond.getDayCountConvention(), bond.getYieldCurve());
	}
	
	public double quotedPrice(Date today) {
		CashFlows CF = getCashFlows();
		Date[] dates = CF.getDates();
		double[] cashflows = CF.getCashFlows();
		
		int i=0;
		while((dates[i].getSerialNumber() < today.getSerialNumber()) && (i < dates.length))
			i++;
		double PV = 0.0;
		if(i == dates.length)
			return 0.0;
		else if(i != (dates.length - 1)) {
			int j;
			double timeToNextCF;
			for(j=i; j<dates.length; j++) {
				timeToNextCF = today.dayCount(dates[j], dayCountConvention);
				//we are calculating discount factor according to continuous compounding
				//a more general method would be to pass an interest composition
				PV += cashflows[j] * yieldCurve.discountFactor(timeToNextCF);
			}
		}
		
		return PV;
	}
	
	public double fairValue(Date today) {
		double PV = quotedPrice(today);
		double accruedInterest;
		
		if((today.getSerialNumber() <= issueDate.getSerialNumber()) || 
				(today.getSerialNumber() >= maturityDate.getSerialNumber()))
			accruedInterest = 0.0;
		else
		{
			CashFlows CF = getCashFlows();
			Date[] dates = CF.getDates();

			int i=0;
			while((dates[i].getSerialNumber() < today.getSerialNumber()) && 
					(i < dates.length))
				i++;
		
			Date interestAccrualDate;
			
			if (today.getSerialNumber() < firstCouponDate.getSerialNumber())
				interestAccrualDate = new Date(issueDate);
			else
				interestAccrualDate = new Date(dates[i-1]);
			
			double numberofdays, numberofdays2, numberofcoupon;
			
			switch(frequency){
			case NoFrequency:
				numberofcoupon = 1.0;//to avoid division by zero
				break;
			case Once:
				numberofcoupon = 1.0;
				break;
			case Annual:
				numberofcoupon = 1.0;
				break;
			case Semiannual:
				numberofcoupon = 2.0;
				break;
			case EveryFourthMonth:
				numberofcoupon = 3.0;
				break;
			case Quarterly:
				numberofcoupon = 4.0;
				break;
			case Bimonthly:
				numberofcoupon = 6.0;
				break;
			case Monthly:
				numberofcoupon = 12.0;
				break;
			default:
				numberofcoupon = 1.0;
				break;
			}
			
			switch (dayCountConvention) {
			case ACT_365:
				numberofdays = (double)(today.getSerialNumber()-interestAccrualDate.getSerialNumber());
				numberofdays2 = 365.0/numberofcoupon;
				break;
			case ACT_360:
				numberofdays = (double)(today.getSerialNumber()-interestAccrualDate.getSerialNumber());
				numberofdays2 = 360.0/numberofcoupon;
				break;
			case Day30_365:
				numberofdays = (double)(today.dayOfMonth()+30-interestAccrualDate.dayOfMonth() +
						30*(12*(today.year()-interestAccrualDate.year()) + 
								today.month().value()-interestAccrualDate.month().value()));
				numberofdays2 = 365.0/numberofcoupon;
				break;
			case Day30_360:
				numberofdays = (double)(today.dayOfMonth()+30-interestAccrualDate.dayOfMonth() + 
						30*(12*(today.year()-interestAccrualDate.year()) + 
								today.month().value()-interestAccrualDate.month().value()));
				numberofdays2 = 360.0/numberofcoupon;
				break;
			default:
				numberofdays = (double)(today.getSerialNumber()-dates[i].getSerialNumber()); 
				numberofdays2 = 365.0/numberofcoupon;
			}
			accruedInterest = numberofdays/numberofdays2*coupon/numberofcoupon*faceAmount;
		}
		
		return PV + accruedInterest;
	}
	
	public double yieldToMaturity(Date today) {
		double price = fairValue(today);
		double yield = 0.05;
		double deriv_price;
		double yield_price;


		CashFlows CF = getCashFlows();
		Date[] dates = CF.getDates();
		double[] cashflows = CF.getCashFlows();

		int i=0;
		while((dates[i].getSerialNumber() < today.getSerialNumber()) && (i < dates.length))
			i++;
		if (i==dates.length)
			return 0;
		
		double timetonextCF;
		int k;
		int j;
		for(k=0;k<100;k++){
			deriv_price = 0;
			yield_price = 0;
			for(j=i; j<dates.length; j++){
				timetonextCF = today.dayCount(dates[j], dayCountConvention);
				yield_price += cashflows[j]*Math.exp(-yield*timetonextCF);
				deriv_price += -timetonextCF*cashflows[j]*Math.exp(-yield*timetonextCF);
			}
			yield += (price - yield_price)/deriv_price;
		}
		return yield;
	}
	
	public double duration(Date today) {
		double yield = yieldToMaturity(today);
		CashFlows CF = getCashFlows();
		Date[] dates = CF.getDates();
		double[] cashflows = CF.getCashFlows();

		int i=0;
		while((dates[i].getSerialNumber() < today.getSerialNumber()) && (i<dates.length))
			i++;
		if (i == dates.length)
			return 0;

		double result = 0;
		double PV = 0;
		double timetonextCF;
		int j;
		for(j=i;j<dates.length;j++){
				timetonextCF = today.dayCount(dates[j], dayCountConvention);
				result += cashflows[j]*Math.exp(-yield*timetonextCF)*timetonextCF;
				PV += cashflows[j]*Math.exp(-yield*timetonextCF);
		}

		result /= PV;
		return result;
	}
	
	public double convexity(Date today) {
		double yield = yieldToMaturity(today);
		CashFlows CF = getCashFlows();
		Date[] dates = CF.getDates();
		double[] cashflows = CF.getCashFlows();

		int i=0;
		while((dates[i].getSerialNumber()<today.getSerialNumber()) && (i<dates.length))
			i++;
		if (i==dates.length) 
			return 0;
		double result = 0;
		double PV = 0;
		double timetonextCF;
		int j;
		for(j=i; j<dates.length; j++){
				timetonextCF = today.dayCount(dates[j], dayCountConvention);
				result += cashflows[j]*Math.exp(-yield*timetonextCF)*timetonextCF*timetonextCF;
				PV += cashflows[j]*Math.exp(-yield*timetonextCF);
		}

		result /= PV;
		return result;
	}
	
	public double quotedPrice() {
		return quotedPrice(issueDate);
	}
	
	public double fairValue() {
		return fairValue(issueDate);
	}
	
	public double yieldToMaturity() {
		return yieldToMaturity(issueDate);
	}
	
	public double duration() {
		return duration(issueDate);
	}
	
	public double convexity() {
		return convexity(issueDate);
	}
	
	public Date getMaturityDate() {
		return new Date(maturityDate);
	}
	
	public Date getIssueDate() {
		return new Date(issueDate);
	}
	
	public Date getFirstCouponDate() {
		return new Date(firstCouponDate);
	}
	
	public double getFaceAmount() {
		return faceAmount;
	}
	
	public double getCoupon() {
		return coupon;
	}
	
	public Frequency getFrequency() {
		return frequency;
	}
	
	public DayCountConvention getDayCountConvention() {
		return dayCountConvention;
	}
	
	public YieldCurve getYieldCurve() {
		return new YieldCurve(yieldCurve);
	}
	
	public CashFlows getCashFlows() {
		double timeToMaturity = firstCouponDate.dayCount(maturityDate, dayCountConvention);
		double numberOfCashFlows;
		double couponRate;
		int numberOfMonthPerPeriod;
		
		switch(frequency) {
		case NoFrequency:
			numberOfCashFlows = 1.0;
			couponRate = 0;
			numberOfMonthPerPeriod = 0;
			break;
		case Once:
			numberOfCashFlows = 1.0;
			couponRate = 0;
			numberOfMonthPerPeriod = 0;
			break;
		case Annual:
			numberOfCashFlows = timeToMaturity + 1;
			couponRate = coupon;
			numberOfMonthPerPeriod = 12;
			break;
		case Semiannual:
			numberOfCashFlows = 2.0*timeToMaturity + 1;
			couponRate = coupon / 2.0;
			numberOfMonthPerPeriod = 6;
			break;
		case EveryFourthMonth:
			numberOfCashFlows = 3.0*timeToMaturity + 1;
			couponRate = coupon / 3.0;
			numberOfMonthPerPeriod = 4;
			break;
		case Quarterly:
			numberOfCashFlows = 4.0*timeToMaturity + 1;
			couponRate = coupon / 4.0;
			numberOfMonthPerPeriod = 3;
			break;	
		case Bimonthly:
			numberOfCashFlows = 6.0*timeToMaturity + 1;
			couponRate = coupon / 6.0;
			numberOfMonthPerPeriod = 2;
			break;
		case Monthly:
			numberOfCashFlows = 12.0*timeToMaturity + 1;
			couponRate = coupon / 12.0;
			numberOfMonthPerPeriod = 1;
			break;
		default:
			numberOfCashFlows = 1.0;
			couponRate = coupon;
			numberOfMonthPerPeriod = 0;
			break;
		}
		
		Date d = new Date(firstCouponDate);
		int numberOfCF = (int)(numberOfCashFlows + 0.1);
		
		Date[] dates = new Date[numberOfCF];
		double[] cashflows = new double[numberOfCF];
		
		dates[0] = new Date(d);
		cashflows[0] = faceAmount*couponRate;
		
		for(int i=1;i<numberOfCF;i++) {
			dates[i] = Date.applyConvention(d.advance(i*numberOfMonthPerPeriod, TimeUnit.Months), BusinessDayConvention.Following);
			cashflows[i] = faceAmount*couponRate;
		}
		if(frequency==Frequency.Once)
			cashflows[0] = faceAmount;
		else
			cashflows[numberOfCF-1] += faceAmount;
		
		CashFlows CF = new CashFlows(dates, cashflows);
		
		return CF;
	}
}
