package rates;

import common.Date;

public class SwapCashFlows {

	private double[] flowAmounts;

	private Date[] flowDates;

	/** Constructor for fixed leg */
	public SwapCashFlows(SwapLeg swapLeg, double fixedRate) {
		flowAmounts = swapLeg.getAmounts();
		flowDates = swapLeg.getDates();
		Date d = Date.setDateToToday();
		
		// in the following calculations we are assuming ACT_365 day count
		// convention. Maybe we should enhance this part by accepting any
		// other day count convention.
		flowAmounts[0] = flowAmounts[0] * fixedRate
				* (flowDates[0].getSerialNumber() - d.getSerialNumber())
				/ 365.0;
		for (int i = 1; i < flowDates.length; i++) {
			flowAmounts[i] = flowAmounts[i]
					* fixedRate
					* (flowDates[i].getSerialNumber() - flowDates[i - 1]
							.getSerialNumber()) / 365.0;
		}
	}

	/** Constructor for floating leg by giving a yield curve object */
	public SwapCashFlows(SwapLeg swapLeg, YieldCurve floatingCurve) {
		flowAmounts = swapLeg.getAmounts();
		flowDates = swapLeg.getDates();
		Date d = Date.setDateToToday();
		
		// in the following calculations we are assuming ACT_365 day count
		// convention. Maybe we should enhance this part by accepting any
		// other day count convention.
		// valuation method: the swap is a ptf of FRAs (see Hull book)
		flowAmounts[0] = flowAmounts[0]
				* floatingCurve.forwardRate(d, flowDates[0],
						InterestComposition.Discrete)
				* (flowDates[0].getSerialNumber() - d.getSerialNumber())
				/ 365.0;
		for (int i = 1; i < flowDates.length; i++) {
			flowAmounts[i] = flowAmounts[i]
					* floatingCurve.forwardRate(flowDates[i - 1], flowDates[i],
							InterestComposition.Discrete)
					* (flowDates[i].getSerialNumber() - flowDates[i - 1]
							.getSerialNumber()) / 365.0;
		}
	}

	/** Get the fair value of the swap : discounted value of cash flows */
	public double getFairValue(YieldCurve yieldCurve) {
		double result = 0.0;
		for (int i = 0; i < flowAmounts.length; i++) {
			result += flowAmounts[i]
					* yieldCurve.discountFactor(flowDates[i],
							InterestComposition.Discrete);
		}
		return result;
	}
}
