package repository;


import java.util.List;
import java.util.Date;

import org.hibernate.*;
import org.hibernate.criterion.*;
import data.*;
import exception.*;
import manager.*;
/**
 * DAO providing basic methods for managing Articleclass-Objects.
 * @author Ferdinand Kaiser
 */
public class ArticleDAO {

	private Session session;

	/**
	 * This constructor is used by the DAOManager, its a factory that initializes all DAOs with the current session.
	 * Note that every session.commit() will close the session, the DAOManager must initialize the DAOs again.
	 * Use a Open Session in View pattern to prevent this.
	 */
	public ArticleDAO(){
		this.session=HibernateUtil.getSessionFactory().getCurrentSession();
	}

	/**
	 * Creates article object and saves it, sets creationDate with actual date and sets
	 * visible property to true
	 * @param owner owner of the article 
	 * @param title title of the article
	 * @param text text of the article
	 * @param image image of the article
	 * @return the article object
	 * @throws MissingValueException if there is a not-null value that is null
	 */
	public Article createAndStore( User owner, String title, String text, Image image)
	throws MissingValueException{
		Article article=new Article(owner, title, new Date(), text, null, true, image);
		try{
			session.save(article);
		}
		catch(PropertyValueException ex){
			throw new MissingValueException("A not-null value is possibly null!");
		}
		return article;
	}

	/**
	 * @return a list of all articles
	 * @throws NoSuchArticleException if there aren't articles
	 */
	public List<Article> getList() throws NoSuchArticleException{
		List<Article> result=session.createQuery("from Article").list();
		if (result.size()==0) throw new NoSuchArticleException();
		return result;
	}

	/**
	 * filters all articles by a given tag
	 * @param name name of the tag
	 * @return a list of articles that are taged with this name
	 * @throws NoSuchTagException if there isn't a tag with this name
	 * @throws NoSuchArticleException if there are no articles tagged with this name
	 */
	public List<Article> getByTag(String name) throws NoSuchTagException,NoSuchArticleException{
		Tag tag=DAOManager.getTagDAO().getByName(name);
		Query query=session.createQuery("from Article a where :tag in elements(a.tags)");
		query.setEntity("tag",tag);
		List<Article> result=query.list();
		if (result==null) throw new NoSuchArticleException();
		return result;
	}

	/**
	 * gets an article by its unique id
	 * @param id unique id
	 * @return article
	 * @throws NoSuchArticleException if there isn't an article with this id
	 */
	public Article getById(int id) throws NoSuchArticleException{
		Article result=(Article)session.get(Article.class,id);
		if (result==null) throw new NoSuchArticleException();
		return result;
	}

	/**
	 * filters all Articles by a given owner
	 * @param owner owner to filter by
	 * @return list of all Articles from this owner
	 * @throws NoSuchEventException if there are no Articles from this owner
	 */
	public List<Article> getByOwner(User owner) throws NoSuchArticleException{
		Criteria criteria = session.createCriteria(Article.class);
		criteria.add(Expression.eq("owner",owner));
		List<Article> result=criteria.list();
		if (result.size()==0) throw new NoSuchArticleException();
		return result;
	}		

	/**
	 *  Retrieves given amount of Articles
	 * @param start sets the first row to retrieve, starting from 0
	 * @param count sets the maximum number of results to retrieve
	 * @return
	 */
	public List<Article> getList(int start, int count)throws NoSuchArticleException{
		Query query=session.createQuery("from Article order by creationDate DESC");
		query.setFirstResult(start);
		query.setMaxResults(count);
		List<Article> result=query.list();
		if (result.size()==0) throw new NoSuchArticleException();
		return result;
	}

}
