package repository;


import java.util.Date;
import java.util.List;
import org.hibernate.*;
import org.hibernate.criterion.*;
import data.*;
import manager.DAOManager;
import exception.*;

/**
 * DAO providing basic methods for managing Commentclass-Objects.
 * @author Ferdinand Kaiser
 *
 */
public class CommentDAO {

	private Session session;

	/**
	 * This constructor is used by the DAOManager, its a factory that initializes all DAOs with the current session.
	 * Note that every session.commit() will close the session, the DAOManager must initialize the DAOs again.
	 * Use a Open Session in View pattern to prevent this.
	 */
	public CommentDAO(){
		this.session=HibernateUtil.getSessionFactory().getCurrentSession();
	}	

	/**
	 * Creates comment object and saves it, sets creationDate with actual date and sets
	 * visible property to true. Assigns comment to the article, referenced by parentID.
	 * @param owner owner the article 
	 * @param title title the article
	 * @param text text the article
	 * @param parentId the id of the article, this comment is associated with
	 * @return the comment object
	 * @throws NoSuchArticleException if the associated Article doesn't exist
	 * @throws MissingValueException if there is a not-null value that is null
	 */
	public Comment createArticleComment(User owner, String title, String text, int parentId) 
	throws NoSuchArticleException, MissingValueException{
		Comment comment=new Comment(owner, title, new Date(), text, null, true);
		Article article=DAOManager.getArticleDAO().getById(parentId);
		article.getComments().add(comment);
		try{
			session.save(comment);
		}
		catch(PropertyValueException ex){
			throw new MissingValueException("A not-null value is possibly null!");
		}
		return comment;
	}

	/**
	 * Creates comment object and saves it, sets creationDate with actual date and sets
	 * visible property to true. Assigns comment to the event, referenced by parentID.
	 * @param owner owner the event
	 * @param title title the event
	 * @param text text the event
	 * @param parentId the id of the event, this comment is associated with
	 * @return the comment object
	 * @throws NoSuchEventException if the associated Event doesn't exist
	 * @throws MissingValueException if there is a not-null value that is null
	 */
	public Comment createEventComment(User owner, String title, String text, int parentId) 
	throws NoSuchEventException, MissingValueException{
		Comment comment=new Comment(owner, title, new Date(), text, null, true);
		Event event=DAOManager.getEventDAO().getById(parentId);
		event.getComments().add(comment);
		try{
			session.save(comment);
		}
		catch(PropertyValueException ex){
			throw new MissingValueException("A not-null value is possibly null!");
		}
		return comment;
	}

	/**
	 * Creates comment object and saves it, sets creationDate with actual date and sets
	 * visible property to true. Assigns comment to the forumthread, referenced by parentID.
	 * @param owner owner the forumthread 
	 * @param title title the forumthread
	 * @param text text the forumthread
	 * @param parentId the id of the forumthread, this comment is associated with
	 * @return the comment object
	 * @throws NoSuchForumThreadException if the associated ForumThread doesn't exist
	 * @throws MissingValueException if there is a not-null value that is null
	 */
	public Comment createForumThreadComment(User owner, String title, String text, int parentId) 
	throws NoSuchForumThreadException,MissingValueException{
		Comment comment=new Comment(owner, title, new Date(), text, null, true);
		ForumThread thread=DAOManager.getForumThreadDAO().getById(parentId);
		thread.getComments().add(comment);
		thread.setLastUpdated(new Date());
		try{
			session.save(comment);
		}
		catch(PropertyValueException ex){
			throw new MissingValueException("A not-null value is possibly null!");
		}
		return comment;		
	}	


	/**
	 * gets an comment by its unique id
	 * @param id
	 * @return comment
	 * @throws NoSuchCommentException if there isn't a comment with this id
	 */
	public Comment getById(Long id) throws NoSuchCommentException{
		Comment result=(Comment)session.get(Article.class,id);
		if (result==null) throw new NoSuchCommentException();
		return result;
	}	

	/**
	 * filters all Comments by a given owner
	 * @param owner owner to filter by
	 * @return list of all Comments from this owner
	 * @throws NoSuchEventException if there are no Comments from this owner
	 */
	public List<Comment> getByOwner(User owner) throws NoSuchCommentException{
		Criteria criteria = session.createCriteria(Comment.class);
		criteria.add(Expression.eq("owner",owner));
		List<Comment> result=criteria.list();
		if (result.size()==0) throw new NoSuchCommentException();
		return result;
	}	

	/**
	 * @return a list of all comments
	 * @throws NoSuchcommentException if there aren't any comments
	 */
	public List<Comment> getList() throws NoSuchCommentException{
		List<Comment> result=session.createQuery("from comment order by creationDate DESC").list();
		if (result.size()==0) throw new NoSuchCommentException();
		return result;
	}	

	/**
	 * Retrieves given amount of comments
	 * @param start sets the first row to retrieve, starting from 0
	 * @param count sets the maximum number of results to retrieve
	 * @return
	 */
	public List<Comment> getList(int start, int count)throws NoSuchCommentException{
		Query query=session.createQuery("from comment order by creationDate DESC");
		query.setFirstResult(start);
		query.setMaxResults(count);
		List<Comment> result=query.list();
		if (result.size()==0) throw new NoSuchCommentException();
		return result;
	}

}
