package repository;

import java.util.*;
import manager.DAOManager;
import org.hibernate.*;
import org.hibernate.criterion.*;
import exception.*;
import data.*;

/**
 * DAO providing basic methods for managing Eventclass-Objects.
 * @author Ferdinand Kaiser, 06.04.08
 */
public class EventDAO {

	private Session session;

	/**
	 * This constructor is used by the DAOManager, its a factory that initializes all DAOs with the current session.
	 * Note that every session.commit() will close the session, the DAOManager must initialize the DAOs again.
	 * Use a Open Session in View pattern to prevent this.
	 */	
	public EventDAO(){
		this.session=HibernateUtil.getSessionFactory().getCurrentSession();
	}

	/**
	 * Creates event object and saves it, sets creationDate with actual date and sets
	 * visible property to true
	 * @param owner owner of the event 
	 * @param title title of the event
	 * @param text text of the event
	 * @param location location of the event
	 * @param image image of the event
	 * @param startDate start date of the event
	 * @param endDate end date of the event 
	 * @return event object
	 * @throws MissingValueException if there is a not-null value that is null	 * 
	 */
	public Event createAndStore(User owner, String title, String text, Location location, Image image, Date startDate, Date endDate)
	throws MissingValueException{
		Event Event=new Event(owner, title, new Date(), text, null, true, location, image, startDate, endDate);
		try{	
			session.save(Event);
		}
		catch(PropertyValueException ex){
			throw new MissingValueException("A not-null value is possibly null!");
		}
		return Event;
	}

	/**
	 * Creates event object and saves it, sets creationDate with actual date and sets
	 * visible property to true
	 * @param owner owner of the event 
	 * @param title title of the event
	 * @param text text of the event
	 * @param location location of the event
	 * @param image image of the event
	 * @param startDate start date of the event
	 * @param endDate end date of the event 
	 * @return event object
	 */
	public Event createAndStore(User owner, String title, String text, Location location, Image image, GregorianCalendar startDate, GregorianCalendar endDate){
		Event Event=new Event(owner, title, new Date(), text, null, true, location, image, startDate.getTime(), endDate.getTime());
		session.save(Event);
		return Event;
	}

	/**
	 * filters all events by a given tag
	 * @param name name of the tag
	 * @return  a list of events that are taged with this name
	 * @throws NoSuchTagException if there isn't a tag with this name
	 * @throws NoSuchEventException if there are no events tagged with this name
	 */
	public List<Event> getByTag(String name) throws NoSuchTagException, NoSuchEventException{
		Tag tag=DAOManager.getTagDAO().getByName(name);	
		Query query=session.createQuery("from Event e where :tag in elements(e.tags)");
		query.setEntity("tag",tag);
		List<Event> result=query.list();
		if (result.size()==0) throw new NoSuchEventException();
		return result;		
	}	

	/**
	 * 
	 * @return a list of all events
	 * @throws NoSuchEventException if there are no events
	 */
	public List<Event> getList() throws NoSuchEventException{
		List<Event> result=session.createQuery("from Event order by startDate DESC").list();
		if (result.size()==0) throw new NoSuchEventException();
		return result;
	}

	/**
	 * gets an events by its unique id
	 * @param id unique id
	 * @return article
	 * @throws NoSuchEventException if there isn't an events with this id
	 */
	public Event getById(int id) throws NoSuchEventException{
		Event result=(Event)session.get(Event.class,id);
		if (result==null) throw new NoSuchEventException();
		return result;
	}

	/**
	 * filters all events by a given month, the year is set to the actual year
	 * @param month month to filter by
	 * @return a list of all events in this month
	 * @throws NoSuchEventException if there are no events in this month
	 */
	public List<Event> getByMonth(int month) throws NoSuchEventException{
		GregorianCalendar startMonth=DateUtil.get();
		GregorianCalendar endMonth=DateUtil.get();
		Criteria criteria = session.createCriteria(Event.class);
		// month-1 because GregorianCalendar starts with month 0
		startMonth.set(startMonth.get(startMonth.YEAR), month-1, 0,0,0);
		endMonth.set(endMonth.get(endMonth.YEAR), month, 0,0,0);

		criteria.add(Restrictions.or(
				Restrictions.and(Expression.ge("endDate",startMonth.getTime()), Expression.le("endDate",endMonth.getTime())),
				Restrictions.and(Expression.ge("startDate",startMonth.getTime()), Expression.lt("startDate",endMonth.getTime()))		
		));
		criteria.addOrder( Order.desc("startDate"));
		List<Event> result=criteria.list();
		if (result.size()==0) throw new NoSuchEventException();
		return result;
	}

	/**
	 * filters all events by a given month and year 
	 * @param month month to filter by
	 * @param year year to filter by
	 * @return a list of all events in this month
	 * @throws NoSuchEventException if there are no events in this month
	 */	
	public List<Event> getByMonth(int month, int year) throws NoSuchEventException{

		GregorianCalendar startMonth=DateUtil.get(0,month,year);
		GregorianCalendar endMonth=DateUtil.get(0,month+1,year);
		Criteria criteria = session.createCriteria(Event.class);
		criteria.add(Restrictions.or(
				Restrictions.and(Expression.ge("endDate",startMonth.getTime()), Expression.lt("endDate",endMonth.getTime())),
				Restrictions.and(Expression.ge("startDate",startMonth.getTime()), Expression.lt("startDate",endMonth.getTime()))		
		));
		criteria.addOrder(Order.desc("startDate"));
		List<Event> result=criteria.list();
		if (result.size()==0) throw new NoSuchEventException();
		return result;
	}	
	
	/**
	 * returns count events that will happen in future
	 * @param count
	 * @return count events that will happen in future
	 * @throws NoSuchEventException if there are no events
	 */
	public List<Event> getUpcoming(int count) throws NoSuchEventException{

		GregorianCalendar startDate=DateUtil.get();
		Criteria criteria = session.createCriteria(Event.class);
		criteria.add(Restrictions.or(
				Expression.ge("endDate",startDate.getTime()),
				Expression.ge("startDate",startDate.getTime())		
		));
		criteria.addOrder(Order.asc("startDate"));
		criteria.setMaxResults(count);
		List<Event> result=criteria.list();
		if (result.size()==0) throw new NoSuchEventException();
		return result;
	}

	/**
	 * returns count events that will happen in future at a specified location
	 * @param count
	 * @param locationId id of the location
	 * @return count events that will happen in future at a specified location
	 * @throws NoSuchEventException
	 */
	public List<Event> getUpcomingByLocation(int count, int locationId) throws NoSuchEventException, NoSuchLocationException{
		Location location=DAOManager.getLocationDAO().getById(locationId);
		GregorianCalendar startDate=DateUtil.get();
		Criteria criteria = session.createCriteria(Event.class);
		criteria.add(Restrictions.or(
				Expression.ge("endDate",startDate.getTime()),
				Expression.ge("startDate",startDate.getTime())		
		));
		criteria.add(Expression.eq("location",location));
		criteria.addOrder(Order.asc("startDate"));
		criteria.setMaxResults(count);
		List<Event> result=criteria.list();
		if (result.size()==0) throw new NoSuchEventException();
		return result;
	}	
	
	
	/**
	 * filters all events by a given day, month and year
	 * @param day day to filter by
	 * @param month month to filter by
	 * @param year year to filter by
	 * @return list of all events at this date
	 * @throws NoSuchEventException if there are no events at this date
	 */
	public List<Event> getByDate(int day, int month, int year) throws NoSuchEventException{
		GregorianCalendar startDay=DateUtil.get(day,month,year);
		GregorianCalendar endDay=DateUtil.get(day+1,month,year);
		Criteria criteria = session.createCriteria(Event.class);
		criteria.add(Expression.le("startDate",endDay.getTime()));
		criteria.add(Expression.ge("endDate",startDay.getTime()));
		criteria.addOrder( Order.asc("startDate"));
		List<Event> result=criteria.list();
		if (result.size()==0) throw new NoSuchEventException();
		return result;
	}		


	/**
	 * filters all events by a given owner
	 * @param owner owner to filter by
	 * @return list of all events from this owner
	 * @throws NoSuchEventException if there are no events from this owner
	 */
	public List<Event> getByOwner(User owner) throws NoSuchEventException{
		Criteria criteria = session.createCriteria(Event.class);
		criteria.add(Expression.eq("owner",owner));
		List<Event> result=criteria.list();
		criteria.addOrder( Order.desc("startDate"));
		if (result.size()==0) throw new NoSuchEventException();
		return result;
	}		
	
	
	/**
	 * filters all events by a given location
	 * @param location location to filter by
	 * @return list of all events from this owner
	 * @throws NoSuchEventException if there are no events from this owner
	 */
	public List<Event> getByLocation(Location location) throws NoSuchEventException{
		Criteria criteria = session.createCriteria(Event.class);
		criteria.add(Expression.eq("location",location));
		List<Event> result=criteria.list();
		criteria.addOrder(Order.asc("startDate"));
		if (result.size()==0) throw new NoSuchEventException();
		return result;
	}	

	/**
	 *  Retrieves given amount of Events
	 * @param start sets the first row to retrieve, starting from 0
	 * @param count sets the maximum number of results to retrieve
	 * @return
	 */
	public List<Event> getList(int start, int count)throws NoSuchEventException{
		Query query=session.createQuery("from Event order by name ASC");
		query.setFirstResult(start);
		query.setMaxResults(count);
		List<Event> result=query.list();
		if (result.size()==0) throw new NoSuchEventException();
		return result;
	}	

}
