package repository;

import javax.activation.MimetypesFileTypeMap;

import repository.FileDAO;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;

import exception.*;
/**
 * This class provides basic methods for storing files.
 * @author Ferdinand Kaiser
 *
 */
public class FileDAO {


	private String baseDir="images/";
	// 25 MiB
	// TODO @Roland Ueberschreiten der maximalen Verzeichnisgroesse im Controller testen !!!
	private long baseDirMaxSize=25;

	private int buffersize=4096;

	/**
	 * 
	 * @param baseDir name of the directory where the files are stored in
	 * @param baseDirMaxSize maximum size of the directory in MiB
	 */
	public FileDAO(String baseDir, long baseDirMaxSize) {
		super();
		this.baseDir = baseDir;
		this.baseDirMaxSize=(long)Math.pow(2,20)*baseDirMaxSize;
	}

	/**
	 * 
	 * @param baseDir baseDir name of the directory where the files are stored in
	 * @param baseDirMaxSize baseDirMaxSize maximum size of the directory in MiB
	 * @param buffersize size of the internal buffer, standard is 4096 Byte
	 */
	public FileDAO(String baseDir, long baseDirMaxSize, int buffersize) {
		super();
		this.baseDir = baseDir;
		this.baseDirMaxSize=(long)Math.pow(2,20)*baseDirMaxSize;
		this.buffersize=buffersize;
	}

	/**
	 * 
	 * @return name of the directory where the files are stored in
	 */
	public  String getBaseDir() {
		return baseDir;
	}

	/**
	 * sets name of the directory where the files are stored in
	 * @param baseDir
	 */
	private void setBaseDir(String baseDir) {
		this.baseDir = baseDir;
	}

	/**
	 * 
	 * @return maximum size of the directory in MiB
	 */
	public long getBaseDirMaxSize() {
		return baseDirMaxSize;
	}

	/**
	 * sets maximum size of the directory in MiB
	 * @param baseDirMaxSize
	 */
	private  void setBaseDirMaxSize(long baseDirMaxSize) {
		this.baseDirMaxSize=(long)Math.pow(2,20)*baseDirMaxSize;
	}

	/**
	 * 
	 * @return internal buffersize
	 */
	public int getBuffersize() {
		return buffersize;
	}

	/**
	 * sets internal buffersize
	 * @param buffersize
	 */
	public void setBuffersize(int buffersize) {
		this.buffersize = buffersize;
	}

	/**
	 * copys a file from one location to another
	 * @param fromFileName path to the sourcefile
	 * @param toFileName path to the targetfile, note that the baseDir string will be attached in front automatically
	 * @throws IOException
	 */
	public void copy(String fromFileName, String toFileName) throws IOException {
		File fromFile = new File(fromFileName);
		File toFile = new File(baseDir+toFileName);

		if (!fromFile.exists())
			throw new NoSuchFileException("FileCopy: " + "no such source file: "+ fromFileName);
		if (!fromFile.isFile())
			throw new NoSuchFileException("FileCopy: " + "can't copy directory: "+ fromFileName +" is a directory.");
		if (!fromFile.canRead())
			throw new NoSuchFileException("FileCopy: " + "source file is unreadable: "+ fromFileName);

		if (toFile.isDirectory()) toFile = new File(toFile, fromFile.getName());

		if (toFile.exists()) {
			if (!toFile.canWrite())
				throw new FileProtectedException("FileCopy: " + "destination file is unwriteable: " + toFileName);
		}
		else{
			String parent = toFile.getParent();
			File dir = new File(parent);
			if (!dir.exists())
				throw new NoSuchDirectoryException("FileCopy: " + "destination directory doesn't exist: " + parent);
			if (dir.isFile())
				throw new NoSuchDirectoryException("FileCopy: "	+ "destination is not a directory: " + parent);
			if (!dir.canWrite())
				throw new NoSuchDirectoryException("FileCopy: "	+ "destination directory is unwriteable: " + parent);		    	


			FileInputStream from = null;
			FileOutputStream to = null;
			try {
				from = new FileInputStream(fromFile);
				to = new FileOutputStream(toFile);
				byte[] buffer = new byte[buffersize];
				int bytesRead;

				while ((bytesRead = from.read(buffer)) != -1)
					to.write(buffer, 0, bytesRead); // write
			} finally {
				if (from != null)
					try {
						from.close();
					} catch (IOException e) {
						e.printStackTrace();
					}
					if (to != null)
						try {
							to.close();
						} catch (IOException e) {
							e.printStackTrace();
						}
			}
		}

	}

	/**
	 * 
	 * @param file
	 * @return the mimetype of the file
	 */
	public static synchronized String getMimeType(File file){
		return new MimetypesFileTypeMap().getContentType(file);
	}

	/**
	 * computes the file of a file or directory
	 * @param file file to check
	 * @return the size of a file or directory
	 */
	public static synchronized long getSize(File file){
		long size=0;
		if (file.isDirectory()){
			File[] files=file.listFiles();
			for(File dummy:files){
				size=size+getSize(dummy);
			}
		}
		else{
			return file.length();
		}
		return size;
	}

	/**
	 * computes the file of a file or directory
	 * @param name path to file or directory to check
	 * @return the size of a file or directory
	 */
	public static synchronized long getSize(String name){
		File file=new File(name);
		return getSize(file);
	}	

}
