package repository;

import java.util.List;
import org.hibernate.Query;
import org.hibernate.Session;
import exception.*;
import data.*;
import org.hibernate.PropertyValueException;

/**
 * DAO providing basic methods for managing Locationclass-Objects.
 * @author Ferdinand Kaiser, 05.05.08
 *
 */
public class LocationDAO {

	private Session session;

	/**
	 * This constructor is used by the DAOManager, its a factory that initializes all DAOs with the current session.
	 * Note that every session.commit() will close the session, the DAOManager must initialize the DAOs again.
	 * Use a Open Session in View pattern to prevent this.
	 */	
	public LocationDAO(){
		this.session=HibernateUtil.getSessionFactory().getCurrentSession();
	}

	/**
	 * Creates location object and saves it
	 * @param name name of the location
	 * @param street streetname of the location
	 * @param postalCode postal code of the location
	 * @param city city the location is situated in
	 * @param description additional description
	 * @param link (http) link
	 * @return location instance
	 * @throws MissingValueException if there is a not-null value that is null	 * 
	 */
	public Location createAndStore(String name, String street, String postalCode, String city,
			String description, String link) throws MissingValueException {
		Location location=new Location(name,street, postalCode, city, description, link);
		try{
			session.save(location);
		}
		catch(PropertyValueException ex){
			throw new MissingValueException("A not-null value is possibly null!");
		}
		return location;
	}

	/**
	 * 
	 * @return a list of all locations
	 * @throws NoSuchLocationException if there aren't any locations
	 */
	public List<Location> getList() throws NoSuchLocationException{
		List<Location> result=session.createQuery("from Location order by name DESC").list();
		if (result.size()==0) throw new NoSuchLocationException();
		return result;
	}	

	/**
	 *  Retrieves given amount of Locations
	 * @param start sets the first row to retrieve, starting from 0
	 * @param count sets the maximum number of results to retrieve
	 * @return
	 */
	public List<Location> getList(int start, int count)throws NoSuchLocationException{
		Query query=session.createQuery("from Location order by name DESC");
		query.setFirstResult(start);
		query.setMaxResults(count);
		List<Location> result=query.list();
		if (result.size()==0) throw new NoSuchLocationException();
		return result;
	}

	/**
	 * gets a location by its unique id
	 * @param id unique id
	 * @return location
	 * @throws NoSuchLocationException if there isn't a location with this id
	 */
	public Location getById(int id) throws NoSuchLocationException{
		Location result=(Location)session.get(Location.class,id);
		if (result==null) throw new NoSuchLocationException();
		return result;

	}

}
