package mks.builder;

import hudson.EnvVars;
import hudson.Extension;
import hudson.Launcher;
import hudson.model.AbstractBuild;
import hudson.model.AbstractProject;
import hudson.model.BuildListener;
import hudson.tasks.BuildStepDescriptor;
import hudson.tasks.Builder;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.logging.Level;
import java.util.logging.Logger;
import mks.MksScm;
import mks.MksUtils;
import mks.cmd.CommandRunner;
import mks.cmd.MksCmdAdd;
import mks.cmd.MksCmdRlog_NonMembers;
import net.sf.json.JSONObject;
import org.apache.commons.lang.StringUtils;
import org.kohsuke.stapler.DataBoundConstructor;
import org.kohsuke.stapler.StaplerRequest;

/**
 *
 * @author James Sheets
 */
public class MksAddFiles 
extends Builder
{
    private final String sandbox;
    private final String issueNumber;
    private final String parameters;
    
    @DataBoundConstructor
    public MksAddFiles(String sandbox, String issueNumber, String parameters)
    {
        this.sandbox = sandbox;
        this.issueNumber = issueNumber;
        this.parameters = parameters;
    }
    
    public String getSandbox() { return sandbox; }
    public String getIssueNumber() { return issueNumber; }
    public String getParameters() { return parameters; }
    
    @Override
    public boolean perform(AbstractBuild<?, ?> build, Launcher launcher, BuildListener listener) 
    throws InterruptedException, IOException
    {
        final PrintStream logger = listener.getLogger();
        boolean success = true;
        
        // Only supported for MKS Source Integrity SCM
        if ( !MksUtils.isUsingMksSandbox(build.getProject().getScm(), logger) )
        {
            return true;
        }
        
        final EnvVars envVars = build.getEnvironment( listener );
        
        // Check to see if it's an environment variable; else use the raw imput
        String envVarKey = MksUtils.stripEnvVarTokens( getSandbox() );
        String sandboxDir = ( envVars.containsKey( envVarKey ) ) 
                            ? envVars.get( envVarKey )
                            : getSandbox();
        
        if (sandboxDir == null)
        {
            listener.getLogger().println("Unable to find sandbox");
            return false;
        }
        
        final MksScm scm = MksScm.class.cast( build.getProject().getScm() );
        final CommandRunner runner = new CommandRunner( listener, build.getWorkspace(), launcher );
        
        
        try {
            final ByteArrayOutputStream rlogOutput = new ByteArrayOutputStream();
            runner.run( new MksCmdRlog_NonMembers(scm.getJobSettings(), sandboxDir), rlogOutput );
            String[] addedFiles = rlogOutput.toString().split("\\r?\\n");
            
            for (String member :addedFiles)
            {
                if (StringUtils.isNotEmpty(member))
                    runner.run( new MksCmdAdd(scm.getJobSettings(), sandboxDir, member, getIssueNumber(), "Hudson add") );
            }
        } catch (Throwable ex) {
            Logger.getLogger(MksAddFiles.class.getName()).log(Level.SEVERE, null, ex);
            success = false;
        }
        
        return success;
    }
    

    @Extension
    public static class DescriptorImpl
    extends BuildStepDescriptor<Builder> 
    {
        @Override
        public boolean isApplicable(Class<? extends AbstractProject> type) 
        {
            // TODO: only if MKS source control type is selected!
            return true;
        }

        @Override
        public String getDisplayName() 
        {
            return "MKS: Add missing files to source";
        }
        
        @Override
        public Builder newInstance(StaplerRequest req, JSONObject formData) 
        throws FormException
        {
            //System.out.println("AddFilesConfig:");
            //System.out.println(formData.toString(2));
            return req.bindJSON(MksAddFiles.class, formData);
        }
        
    }
}
