package mks.changelog;

import hudson.model.AbstractBuild;
import hudson.scm.ChangeLogParser;
import hudson.scm.ChangeLogSet;
import hudson.scm.NullChangeLogParser;
import hudson.util.Digester2;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map.Entry;
import org.apache.commons.digester.Digester;
import org.xml.sax.SAXException;

/**
 * Reads in a change log xml file, and converts it to an instance of our
 * MksChangeLogEntry classes
 * 
 * @author James Sheets
 */
public class MksChangeLogParser
extends ChangeLogParser
{

    @Override
    public ChangeLogSet<? extends Entry> parse( AbstractBuild build, File changelogFile )
    throws IOException, SAXException
    {
        List<MksChangeLogEntry> changeSetList = new ArrayList<MksChangeLogEntry>();
        Digester digester = new Digester2();
        digester.push( changeSetList );

        // When digester reads a {{<modification>}} node it will create a {{SourceIntegrityChangeLogEntry}} object
        digester.addObjectCreate( "*/modification", MksChangeLogEntry.class );
        digester.addSetProperties( "*/modification" );
        // Reads the child node {{<comment>}} and uses {{SourceIntegrityChangeLogEntry.setComment()}} to set the value
        digester.addBeanPropertySetter( "*/modification/action" );
        digester.addBeanPropertySetter( "*/modification/member" );
        digester.addBeanPropertySetter( "*/modification/revision" );
        // Reading the {{<date>}} child node will use the {{SourceIntegrityChangeLogEntry.setDateStr()}} method
        // instead of the default {{SourceIntegrityChangeLogEntry.setDate()}}
        digester.addBeanPropertySetter( "*/modification/date", "dateStr" );
        digester.addBeanPropertySetter( "*/modification/cpid" );
        digester.addBeanPropertySetter( "*/modification/cpDesc" );
        digester.addBeanPropertySetter( "*/modification/mksuser" );
        digester.addBeanPropertySetter( "*/modification/msg" );
        // The digested node/change set is added to the list through {{List.add()}}
        digester.addSetNext( "*/modification", "add" );

        // Do the actual parsing
        FileReader reader = new FileReader( changelogFile );
        digester.parse( reader );
        reader.close();
        ChangeLogSet returnSet = new MksChangeLogSet( build, changeSetList );

        // If there were no changes, this will return an empty changelog set that's
        // recognized by Hudson as truely being empty by
        if ( returnSet.isEmptySet() )
            returnSet = new NullChangeLogParser().parse( build, changelogFile );
        
        return returnSet;
    }
    
    
    public ChangeLogSet<? extends Entry> parse( AbstractBuild build, String changeLog )
    throws IOException, SAXException
    {
        InputStream is = new ByteArrayInputStream(changeLog.getBytes("UTF-8"));
        
        List<MksChangeLogEntry> changeSetList = new ArrayList<MksChangeLogEntry>();
        Digester digester = new Digester2();
        digester.push( changeSetList );

        // When digester reads a {{<modification>}} node it will create a {{SourceIntegrityChangeLogEntry}} object
        digester.addObjectCreate( "*/modification", MksChangeLogEntry.class );
        digester.addSetProperties( "*/modification" );
        // Reads the child node {{<comment>}} and uses {{SourceIntegrityChangeLogEntry.setComment()}} to set the value
        digester.addBeanPropertySetter( "*/modification/action" );
        digester.addBeanPropertySetter( "*/modification/member" );
        digester.addBeanPropertySetter( "*/modification/revision" );
        // bind our input Date string to the setMksDateStr method
        digester.addBeanPropertySetter( "*/modification/date", "mksDateStr" );
        digester.addBeanPropertySetter( "*/modification/cpid" );
        digester.addBeanPropertySetter( "*/modification/cpDesc" );
        digester.addBeanPropertySetter( "*/modification/mksuser" );
        digester.addBeanPropertySetter( "*/modification/msg" );
        // The digested node/change set is added to the list through {{List.add()}}
        digester.addSetNext( "*/modification", "add" );

        // Do the actual parsing
        digester.parse( is );
        ChangeLogSet returnSet = new MksChangeLogSet( build, changeSetList );
        
        return returnSet;
    }

}