package mks.changelog;

import hudson.FilePath;
import hudson.Util;
import hudson.model.AbstractBuild;
import hudson.scm.ChangeLogSet;
import org.kohsuke.stapler.export.ExportedBean;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import nu.xom.*;

import mks.MksUtils;
import mks.config.JobSettings;
import mks.config.Project;


/**
 * Iterable class that contains a set of MksChangeLogEntry's
 * Has the ability to "serialize" itself in an XML format which
 * we can read in later to display them to the user on demand
 * 
 * @author James Sheets
 */
@ExportedBean(defaultVisibility = 999)
public class MksChangeLogSet
extends ChangeLogSet<MksChangeLogEntry>
{

    private List<MksChangeLogEntry> changes = null;


    protected MksChangeLogSet( AbstractBuild<?, ?> build, List<MksChangeLogEntry> changeEntries )
    {
        super( build );
        changes = Collections.unmodifiableList( changeEntries );
    }


    protected MksChangeLogSet( AbstractBuild<?, ?> build )
    {
        super( build );
        changes = new ArrayList<MksChangeLogEntry>();
    }


    @Override
    public boolean isEmptySet()
    {
        return changes.isEmpty();
    }


    public Iterator<MksChangeLogEntry> iterator()
    {
        return changes.iterator();
    }


    public void add( MksChangeLogEntry entry )
    {
        changes.add( entry );
    }


    public MksChangeLogEntry getLast()
    {
        return isEmptySet() ? null : changes.get( changes.size() - 1 );
    }


    /**
     * Save change log entries to XML
     * @param changeLogFile
     * @param workspace
     * @param jobSettings
     * @param projNumber
     * @throws IOException
     * @throws InterruptedException 
     */
    public void save(File changeLogFile, FilePath workspace, JobSettings jobSettings, int projNumber)
    throws IOException, InterruptedException
    {
        // When I try to use Java 1.5+ SAX/DOM libs, I get conflicts with the Xalan
        // library included with testing harness.  Using XOM library instead
        final Project project = jobSettings.getProjects().get(projNumber);
        final String configPath = project.getConfigPath();
        String sandboxName = project.getSandboxName() + "/";
        
        try 
        {
            Builder xmlBuilder = new Builder();
            Document doc = ( changeLogFile.exists() )
                        ? xmlBuilder.build( changeLogFile )
                        : new Document( new Element("changelog") );

            Element root = doc.getRootElement();
            
            Element sandbox = new Element("sandbox");
            root.appendChild(sandbox);
            sandbox.addAttribute( new Attribute("name", Util.xmlEscape( sandboxName )) );
            sandbox.addAttribute( new Attribute("configPath", Util.xmlEscape( configPath )) );

            for (MksChangeLogEntry changeEntry : this)
            {
                Element mod = new Element( "modification" );
                sandbox.appendChild( mod );

                Element action = new Element( "action" );
                action.appendChild( Util.xmlEscape(changeEntry.getAction()) );
                mod.appendChild( action );

                Element member = new Element( "member" );
                member.appendChild( Util.xmlEscape(sandboxName + changeEntry.getMember().replace("\\","/")) );
                mod.appendChild( member );

                Element revision = new Element( "revision" );
                revision.appendChild( Util.xmlEscape(changeEntry.getRevision()) );
                mod.appendChild( revision );

                Element date = new Element( "date" );
                date.appendChild( MksUtils.formatRlogOutputDate( changeEntry.getDate() ) );
                mod.appendChild( date );

                Element cpid = new Element( "cpid" );
                cpid.appendChild( Util.xmlEscape(changeEntry.getCpid()) );
                mod.appendChild( cpid );

                Element mksuser = new Element( "mksuser" );
                mksuser.appendChild( Util.xmlEscape(changeEntry.getMksuser()) );
                mod.appendChild( mksuser );

                Element msg = new Element( "msg" );
                msg.appendChild( Util.xmlEscape(changeEntry.getMsg()) );
                mod.appendChild( msg );

                Element cpDesc = new Element( "cpDesc" );
                cpDesc.appendChild( Util.xmlEscape(changeEntry.getCpDesc()) );
                mod.appendChild( cpDesc );
            }

            // TODO: find a way to write as UTF-8 even though it takes an OutputStream?
            Serializer serializer = new Serializer(new FileOutputStream( changeLogFile.getAbsolutePath() ), "ISO-8859-1");
            serializer.setIndent(4);
            serializer.write(doc);  

        } catch (ParsingException ex) {
            Logger.getLogger(MksChangeLogSet.class.getName()).log(Level.SEVERE, null, ex);
        }

    }

}
