package mks.config;

import hudson.FilePath;
import hudson.model.TaskListener;
import hudson.util.IOUtils;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.lang.StringUtils;

/**
 * Write out specific settings from the last run to a file, so we can compare
 * them with current settings.  Necessary so we can tell if something's changed
 * (project config path) which would require the project to be dropped/recreated.
 * 
 * @author James Sheets
 */
public class PreviousRunSettings
{
    /**
     * Denote a configuration change to the job which would require
     * the sandboxes to be dropped
     */
    public enum CONFIG_CHANGE { NONE, WORKSPACE, CONFIGPATHS };
    
    private JobSettings jobSettings;
    private FilePath workspace;
    private TaskListener listener;


    public PreviousRunSettings( JobSettings jobSettings, FilePath workspace, TaskListener listener )
    {
        this.jobSettings = jobSettings;
        this.workspace = workspace;
        this.listener = listener;
    }
    

    /**
     * Write out our project configuration paths to a file
     * @return If the settings were saved
     */
    public boolean saveConfig()
    {
        listener.getLogger().println("Saving current configuration paths to disk");

        boolean success = false;
        final File configPathFile =
                new File( workspace.getParent() + File.separator + "ConfigPaths.txt" );
        PrintWriter writer = null;
        try
        {
            writer = new PrintWriter( new FileWriter( configPathFile ) );
            writer.println( workspace.getRemote() );
            for ( Project project : jobSettings.getProjects() )
            {
                writer.print( project.getConfigPath() );
                
                // If we're not checking out the tip, also write out the revision number
                if (!project.getBuildType().equals(MksBuildTypeListBoxModel.TIP))
                {
                    writer.print("\t");
                    writer.print(project.getProjectRevision());
                }
                
                writer.println();
            }
            success = true;
        }
        catch ( IOException ex )
        {
            listener.error( ex.getMessage() );
        }

        IOUtils.closeQuietly( writer );
        listener.getLogger().println();
        return success;
    }


    /**
     * Read in our configuration paths from the previous build, and compare
     * them to our current job settings.  Don't know how to serialize a variable
     * I don't want users to configure on the screen, hence writing to a file
     * @return Changes made to the configuration paths of our projects since the last run
     */
    public CONFIG_CHANGE getConfigChanges()
    {
        CONFIG_CHANGE hasChanges = CONFIG_CHANGE.NONE;

        final File configPathFile =
                new File( workspace.getParent() + File.separator + "ConfigPaths.txt" );
        BufferedReader reader = null;
        if ( configPathFile.exists() )
        {
            List<String> previousSavedPaths = new ArrayList<String>();
            List<Project> projects = jobSettings.getProjects();

            try
            {
                reader = new BufferedReader( new FileReader( configPathFile ) );

                // Get the first line - the workspace
                String line = StringUtils.trimToEmpty( reader.readLine() );

                // Workspace has changed, so our sandbox doesn't match the
                // registered path for the sandbox anymore
                if ( !line.equalsIgnoreCase( workspace.getRemote() ) )
                {
                    hasChanges = CONFIG_CHANGE.WORKSPACE;
                }
                else
                {
                    while ((line = StringUtils.trimToNull( reader.readLine() )) != null)
                    {
                        previousSavedPaths.add( line );
                    }

                    // There's been a change to the number sandbox config paths
                    if ( previousSavedPaths.size() != projects.size() )
                    {
                        hasChanges = CONFIG_CHANGE.CONFIGPATHS;
                    }
                    
                    else if ( previousSavedPaths.size() > 0 )
                    {
                        for ( int i=0; i < previousSavedPaths.size(); i++ )
                        {
                            // Found a different between lines.  Could happen if we 
                            // re-arrange the order of the projects, or change 
                            // the config path, or the project revision changes
                            if ( !previousSavedPaths.get(i).equalsIgnoreCase( projects.get(i).getConfigPath() ))
                            {
                                hasChanges = CONFIG_CHANGE.CONFIGPATHS;
                                break;
                            }
                        }
                    }
                }
            }
            catch ( IOException ex )
            {
                listener.error( ex.getMessage() );
            }

            IOUtils.closeQuietly( reader );
        }
        return hasChanges;
    }
    
}
