package mks.publisher;

import hudson.EnvVars;
import hudson.Extension;
import hudson.FilePath;
import hudson.Launcher;
import hudson.Util;
import hudson.model.AbstractBuild;
import hudson.model.AbstractProject;
import hudson.model.BuildListener;
import hudson.model.Result;
import hudson.tasks.BuildStepDescriptor;
import hudson.tasks.BuildStepMonitor;
import hudson.tasks.Publisher;

import hudson.tasks.Recorder;
import hudson.util.FormValidation;
import hudson.util.ListBoxModel;
import java.io.IOException;
import java.io.PrintStream;
import javax.servlet.ServletException;
import mks.MksScm;
import mks.MksUtils;
import mks.cmd.CommandRunner;
import mks.cmd.MksCmdCheckpoint;
import mks.cmd.MksCmdThaw;
import mks.config.MksBuildTypeListBoxModel;
import mks.config.Project;
import org.apache.commons.lang.StringUtils;
import org.kohsuke.stapler.DataBoundConstructor;
import org.kohsuke.stapler.QueryParameter;

/**
 * Post-build actions to thaw and checkpoint projects
 * 
 * @author James Sheets
 */
public class MksRecorder
extends Recorder
{

    private String when;
    private String checkpointDescription;
    private String label;
    private Boolean labelMembers;
    private Boolean checkpoint;


    @DataBoundConstructor
    public MksRecorder(Boolean checkpoint, String when, String checkpointDescription, String label, Boolean labelMembers)
    {
        this.when = when;
        this.checkpointDescription = checkpointDescription;
        this.label = label;
        this.labelMembers = labelMembers;
        this.checkpoint = checkpoint;
    }


    public String getWhen()
    {
        return when;
    }


    public Boolean getCheckpoint()
    {
        return checkpoint;
    }


    public String getLabel()
    {
        return label;
    }


    public String getCheckpointDescription()
    {
        return checkpointDescription;
    }


    public Boolean getLabelMembers()
    {
        return labelMembers;
    }
    

    /**
     * @return Don't allow concurrent builds
     */
    public BuildStepMonitor getRequiredMonitorService()
    {
        return BuildStepMonitor.BUILD;
    }


    @Override
    public boolean perform( final AbstractBuild<?, ?> build, final Launcher launcher, final BuildListener listener )
    throws InterruptedException
    {
        
        boolean success = true;
        final PrintStream logger = listener.getLogger();

        // Only supported for MKS Source Integrity SCM
        if ( !MksUtils.isUsingMksSandbox(build.getProject().getScm(), logger) )
        {
            return true;
        }

        final MksScm scm = MksScm.class.cast( build.getProject().getScm() );
        final FilePath workspace = build.getWorkspace();
        final CommandRunner runner = new CommandRunner( listener, workspace, launcher );

        // Checkpoint each project and thaw if necessary
        int loopCounter = 0;
        for (Project project : scm.getJobSettings().getProjects())
        {
            final FilePath sandboxDir = MksUtils.getSandboxLocation(workspace, project);
            try
            {
                final EnvVars envVars = build.getEnvironment( listener );

                // Substitute any env vars in the description
                String fixedDescription = Util.replaceMacro( Util.fixEmptyAndTrim( getCheckpointDescription() ), envVars );
                // Substitute any env vars in the label
                String fixedLabel = Util.replaceMacro( Util.fixEmptyAndTrim( getLabel() ), envVars );

                if ( !project.getBuildType().equals(MksBuildTypeListBoxModel.PROJECT_REVISION)
                && project.getCheckpoint()
                && validStatusForCheckpointing(build, fixedLabel) )
                {
                    runner.run(
                        new MksCmdCheckpoint(
                            scm.getJobSettings(), sandboxDir.getRemote(), fixedDescription, fixedLabel, labelMembers ) );
                }
            }
            catch(Throwable th)
            {
                listener.error( th.getMessage() );
                success = false;
            }
            finally
            {
                if ( scm.getJobSettings().getFreeze() && project.isFrozen() )
                {
                    try
                    {
                        runner.run( new MksCmdThaw(scm.getJobSettings(), loopCounter) );
                    }
                    catch ( Throwable _ )
                    {
                        success = false;
                        listener.error( String.format("Unable to thaw %s project during post-build", project.getSandboxName()) );
                    }
                }
                loopCounter++;
            }
        }
        
        return success;
    }


    /**
     * Check to see if the project & settings are properly configured so that we
     * should perform a checkpoint
     * 
     * @param build
     * @param cpLabel
     * @return If we can perform a checkpoint
     */
    public boolean validStatusForCheckpointing(final AbstractBuild<?, ?> build, String cpLabel)
    {
        if (getCheckpoint() == false
        || StringUtils.isBlank(cpLabel))
        {
            return false;
        }
        else if(getWhen().equalsIgnoreCase(MksCheckpointWhenListBoxModel.ALWAYS))
        {
            return true;
        }
        else if (getWhen().equalsIgnoreCase( MksCheckpointWhenListBoxModel.SUCCESS)
        && build.getResult() == Result.SUCCESS)
        {
            return true;
        }
        else if (getWhen().equalsIgnoreCase( MksCheckpointWhenListBoxModel.UNSTABLE)
        && build.getResult() == Result.UNSTABLE)
        {
            return true;
        }
        else if (getWhen().equalsIgnoreCase( MksCheckpointWhenListBoxModel.SUCCESS_UNSTABLE)
        && (build.getResult() == Result.SUCCESS || build.getResult() == Result.UNSTABLE))
        {
            return true;
        }
        else
        {
            return false;
        }
    }


    @Extension
    public static final class DescriptorImpl
    extends BuildStepDescriptor<Publisher>
    {

        @Override
        public String getDisplayName()
        {
            return "MKS Source Integrity Post Build Options";
        }
        

        @Override
        public boolean isApplicable(Class<? extends AbstractProject> jobType)
        {
            return true;
        }


        public ListBoxModel doFillWhenItems()
        {
            return new MksCheckpointWhenListBoxModel();
        }


        public FormValidation doCheckCheckpointDescription(@QueryParameter String value)
        throws IOException, ServletException
        {
            if (StringUtils.isBlank(value))
            {
                return FormValidation.error( "Checkpoint description is required" );
            }
            if (value.indexOf("-") > 0)
            {
                return FormValidation.error( "Use of hyphens not permitted." );
            }
            if (value.indexOf( ":" ) > 0)
            {
                return FormValidation.error( "Use of colons not permitted." );
            }
            if (value.indexOf( "[" ) > 0
            || value.indexOf( "]" ) > 0)
            {
                return FormValidation.error( "Use of square brackets not permitted." );
            }
            return FormValidation.ok();
        }


        public FormValidation doCheckLabel(@QueryParameter String value)
        throws IOException, ServletException
        {
            if (value.indexOf("-") > 0)
            {
                return FormValidation.error( "Use of hyphens not permitted." );
            }
            if (value.indexOf( ":" ) > 0)
            {
                return FormValidation.error( "Use of colons not permitted." );
            }
            if (value.indexOf( "[" ) > 0
            || value.indexOf( "]" ) > 0)
            {
                return FormValidation.error( "Use of square brackets not permitted." );
            }
            return FormValidation.ok();
        }

    }
    
}
