package hudson.plugins.si;

import org.apache.commons.io.output.ByteArrayOutputStream;
import hudson.FilePath;
import hudson.Launcher;
import hudson.Launcher.ProcStarter;
import hudson.model.StreamBuildListener;
import hudson.model.TaskListener;
import hudson.util.ForkOutputStream;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.net.ConnectException;
import java.util.concurrent.TimeUnit;
import javax.naming.TimeLimitExceededException;
import org.apache.commons.lang.StringUtils;
import org.apache.tools.ant.util.FileUtils;


/*
 * @author Michael Rack
 */
public class SourceIntegrityLauncher
{

    private final TaskListener listener;
    private final FilePath workspace;
    private final Launcher launcher;
    private final static String INVALID_PROJECT_SEARCH = "is not registered with the system";
    private final static String TIMEOUT_SEARCH = "Timeout after";
    private final static String NO_CURRENT_USER_SEARCH = "Cannot ask for user input in batch mode";
    /**
     * Occassionally, communication with MKS via commandline seems to stop until we log out and back in.  Cause unknown.
     */
    private final static String CONNECTION_REFUSED_SEARCH = "Could not establish server connection";
    /**
     * Authorization failed during login
     */
    private final static String AUTHORIZATION_FAILED_SEARCH = "Authorization failed";


    /**
     * Constructor
     * @param listener
     * @param workspace
     * @param launcher
     */
    public SourceIntegrityLauncher( final TaskListener listener, final FilePath workspace, final Launcher launcher )
    {
        this.listener = listener;
        this.workspace = workspace;
        this.launcher = launcher;
    }


    public TaskListener getListener()
    {
        return listener;
    }


    public FilePath getWorkspace()
    {
        return workspace;
    }


    public Launcher getLauncher()
    {
        return launcher;
    }


    private void checkForErrors( String capturedOutput )
    throws InvalidSandboxException, TimeLimitExceededException, NoCurrentUserException, ConnectException
    {
        // Check for nonexistant user
        boolean nouser = StringUtils.containsIgnoreCase( capturedOutput, NO_CURRENT_USER_SEARCH );
        if ( nouser )
        {
            throw new NoCurrentUserException( "No current user logged into MKS GUI tool" );
        }

        // Check for a timeout
        boolean timeout = StringUtils.containsIgnoreCase( capturedOutput, TIMEOUT_SEARCH );
        if ( timeout )
        {
            throw new TimeLimitExceededException( capturedOutput );
        }

        // Parse output to determine if this is an invalid project
        boolean invalidProject = StringUtils.containsIgnoreCase( capturedOutput, INVALID_PROJECT_SEARCH );
        if ( invalidProject )
        {
            throw new InvalidSandboxException( "Unregistered sandbox" );
        }

        boolean connectionRefused = StringUtils.containsIgnoreCase( capturedOutput, CONNECTION_REFUSED_SEARCH );
        boolean authFailed = StringUtils.containsIgnoreCase( capturedOutput, AUTHORIZATION_FAILED_SEARCH );
        if ( connectionRefused || authFailed )
        {
            throw new ConnectException( capturedOutput );
        }
    }


    public void run( final SourceIntegrityCommand cmd, final InputStream in, final OutputStream out, final int timeoutInSeconds )
    throws IOException, InterruptedException, InvalidSandboxException, TimeLimitExceededException, NoCurrentUserException
    {
        final PrintStream listenerLogger = getListener().getLogger();
        final ForkOutputStream fos = new ForkOutputStream( out, listenerLogger );
        final StreamBuildListener lis = new StreamBuildListener( fos );

        ProcStarter runit = getLauncher().launch()
                .cmds( cmd.getCommand() )
                .masks( cmd.toMaskArray() )
                .pwd( getWorkspace() )
                .stdout( fos );

        // Capture everything to listener and out
        int result = ( timeoutInSeconds <= 0 )
                ? runit.join()
                : runit.start().joinWithTimeout( timeoutInSeconds, TimeUnit.SECONDS, lis );
        
        checkForErrors( out.toString() );
        
        //if ( result != 0)
        //    throw new Exception( "Bad result code returned" );
    }


    /**
     * Run a command without any output - except for the command being run itself - printed to our console
     * @param cmd 
     * @param timtoutInSeconds if 0 or negative, there is no timeout.  Otherwise, specifies the max wait time in seconds
     * @throws IOException
     * @throws InterruptedException
     * @throws InvalidSandboxException This must be a command that uses a --project switch in order to be thrown
     * @throws TimeLimitExceededException If timeoutInSecods is greater than 0, and we've exceeded its limit
     */
    public void runSilent( final SourceIntegrityCommand cmd, int timeoutInSeconds )
    throws IOException, InterruptedException, InvalidSandboxException, TimeLimitExceededException, NoCurrentUserException
    {
        final ByteArrayOutputStream out = new ByteArrayOutputStream();
        final StreamBuildListener lis = new StreamBuildListener( out );

        ProcStarter runit = getLauncher().launch()
                .cmds( cmd.getCommand() )
                .masks( cmd.toMaskArray() )
                .pwd( getWorkspace() )
                .stdout( out );

        // Capture everythingn to out
        int result = ( timeoutInSeconds <= 0 )
                ? runit.join()
                : runit.start().joinWithTimeout( timeoutInSeconds, TimeUnit.SECONDS, lis );

        checkForErrors( out.toString() );

        FileUtils.close( out );
        //if ( result != 0)
        //    throw new Exception( "Bad result code returned" );
    }
}
