package hudson.plugins.si;

import hudson.EnvVars;
import hudson.FilePath;
import hudson.Launcher;
import hudson.Util;
import hudson.model.AbstractBuild;
import hudson.model.BuildListener;
import hudson.model.Descriptor;
import hudson.model.Result;
import hudson.tasks.BuildStepMonitor;
import hudson.tasks.Publisher;
import hudson.util.FormFieldValidator;

import java.io.IOException;
import java.io.PrintStream;

import javax.servlet.ServletException;

import org.apache.commons.io.output.ByteArrayOutputStream;
import org.kohsuke.stapler.StaplerRequest;
import org.kohsuke.stapler.StaplerResponse;

/*
 * @author Michael Rack
 */

public class SourceIntegrityPublisher
extends Publisher
{

    public static final DescriptorImpl DESCRIPTOR = new DescriptorImpl();
    private String _checkpointDescription;
    private String _label;
    private boolean _labelMembers;
    private boolean _thaw = false;
    private boolean _checkpoint;
	

    public SourceIntegrityPublisher() { }


    public boolean isCheckpoint()
    {
        return _checkpoint;
    }
    

    public void setCheckpoint(boolean checkpoint)
    {
        _checkpoint = checkpoint;
    }


    public String getLabel()
    {
        return _label;
    }


    public void setLabel( String label )
    {
        _label = label;
    }


    public String getCheckpointDescription()
    {
        return _checkpointDescription;
    }


    public void setCheckpointDescription( String desc )
    {
        _checkpointDescription = desc;
    }


    public void setLabelMembers( boolean lblmembers )
    {
        _labelMembers = lblmembers;
    }


    public boolean getLabelMembers()
    {
        return _labelMembers;
    }


    /*public boolean getThaw()
    {
        return _thaw;
    }


    public void setThaw( boolean thaw )
    {
        _thaw = thaw;
    }*/


    /**
     * Perform the checkpoint
     * @param build
     * @param launcher
     * @param listener
     * @return
     * @throws java.lang.InterruptedException
     */
    @Override
    public boolean perform( final AbstractBuild<?, ?> build, final Launcher launcher, final BuildListener listener )
    throws InterruptedException
    {
        final PrintStream logger = listener.getLogger();

        // Only supported for MKS Source Integrity SCM
        if ( !(build.getProject().getScm() instanceof SourceIntegrityScm) )
        {
            logger.println( "SourceIntegrity plugin doesn't support labeling for VCS: "
                    + build.getProject().getScm().toString() + "." );
            return true;
        }

        final SourceIntegrityScm scm = SourceIntegrityScm.class.cast( build.getProject().getScm() );
        final FilePath workspace = build.getWorkspace();
        final SourceIntegrityLauncher siLauncher = new SourceIntegrityLauncher( listener, workspace, launcher );

        for (String configPath : scm.getPaths())
        {
            final FilePath projSandbox = scm.getSandboxWorkspace( workspace, configPath );

            try
            {
                final EnvVars envVars = build.getEnvironment( listener );

                // Substitute any env vars in the description
                String fixedDescription = Util.replaceMacro( Util.fixEmptyAndTrim( getCheckpointDescription() ), envVars );

                // Substitute any env vars in the label
                String fixedLabel = Util.replaceMacro( Util.fixEmptyAndTrim( getLabel() ), envVars );

                // Perform checkpoint if label is filled in
                if ( isCheckpoint()
                && fixedLabel != null
                && build.getResult() == Result.SUCCESS )
                {
                    SourceIntegrityCheckpointCommand checkpointCommand = new SourceIntegrityCheckpointCommand(
                            scm.getServer(), scm.getPort(), scm.getUsername(), scm.getPassword(),
                            projSandbox.getRemote(), fixedDescription, fixedLabel, getLabelMembers() );
                    siLauncher.run( checkpointCommand, null, new ByteArrayOutputStream(), 60 * 60 );
                    logger.println( "Successfully checkpointed project: " + configPath );

                }
            }
            catch (Throwable th)
            {
                listener.error( th.getMessage() );
                if ( build.getResult() != Result.FAILURE )
                    build.setResult( Result.UNSTABLE );
            }
            finally
            {
                // TODO: see if we can get/set the isFrozen property from the SCM, and only thaw if its true?
                if ( scm.getFreeze() )
                {
                    try
                    {
                        scm.thawProject( siLauncher, configPath );
                    }
                    catch ( Throwable _ )
                    {
                        if ( build.getResult() != Result.FAILURE )
                        {
                            build.setResult( Result.UNSTABLE );
                        }
                    }
                }
            }
        }

        return true;
    }


    @Override
    public Descriptor<Publisher> getDescriptor()
    {
        return DESCRIPTOR;
    }


    /**
     * @return Don't allow concurrent builds
     */
    public BuildStepMonitor getRequiredMonitorService()
    {
        return BuildStepMonitor.BUILD;
    }


    public static final class DescriptorImpl
    extends Descriptor<Publisher>
    {

        public DescriptorImpl()
        {
            super( SourceIntegrityPublisher.class );
            load();
        }


        protected DescriptorImpl( final Class<? extends Publisher> clazz )
        {
            super( clazz );
        }


        @Override
        public String getDisplayName()
        {
            return "MKS Source Integrity Post-Build Actions";
        }


        @Override
        public String getHelpFile()
        {
            return "/plugin/si/help-publish.html";
        }


        @Override
        public Publisher newInstance( final StaplerRequest req )
        {
            SourceIntegrityPublisher pub = new SourceIntegrityPublisher();
            req.bindParameters( pub, "sip." );
            return pub;
        }


        /**
         * If Checkpointing is enabled, make sure all required fields are set correctly
         * @param req
         * @param rsp
         * @throws IOException
         * @throws ServletException
         */
        public void doValidCheckpointCheck( StaplerRequest req, StaplerResponse rsp )
        throws IOException, ServletException
        {
            new FormFieldValidator( req, rsp, true )
            {
                protected void check()
                throws IOException, ServletException
                {
                    String label = Util.nullify( request.getParameter( "value" ) );
                    if ( null == label ) {
                        errorWithMarkup( "Checkpoint description is required" );
                        return;
                    }
                    label = label.trim();
                    if ( 0 == label.length() ) {
                        errorWithMarkup( "Checkpoint description is required" );
                        return;
                    }
                    if ( label.indexOf( "-" ) > 0 ) {
                        errorWithMarkup( "Use of hyphens not permitted." );
                        return;
                    }
                    if ( label.indexOf( ":" ) > 0 ) {
                        errorWithMarkup( "Use of colons not permitted." );
                        return;
                    }
                    if ( label.indexOf( "[" ) > 0 ) {
                        errorWithMarkup( "Use of square brackets not permitted." );
                        return;
                    }
                    if ( label.indexOf( "]" ) > 0 ) {
                        errorWithMarkup( "Use of square brackets not permitted." );
                        return;
                    }
                    ok();
                }
            }.process();
        }


        /**
         * Ensure label is formatted correctly
         * @param req
         * @param rsp
         * @throws IOException
         * @throws ServletException
         */
        public void doValidLabelCheck( final StaplerRequest req, final StaplerResponse rsp )
        throws IOException, ServletException
        {
            new FormFieldValidator( req, rsp, true )
            {
                @Override
                protected void check()
                throws IOException, ServletException
                {
                    // syntax check first
                    String label = Util.nullify( request.getParameter( "value" ) );
                    if ( null == label ) {
                        ok(); //not entered yet
                        return;
                    }
                    // remove unneeded whitespaces
                    label = label.trim();
                    if ( 0 == label.length() ) {
                        ok(); //just spaces treat as not entered
                        return;
                    }
                    if ( label.indexOf( "-" ) > 0 ) {
                        errorWithMarkup( "Use of hyphens not permitted." );
                        return;
                    }
                    if ( label.indexOf( ":" ) > 0 ) {
                        errorWithMarkup( "Use of colons not permitted." );
                        return;
                    }
                    if ( label.indexOf( "[" ) > 0 ) {
                        errorWithMarkup( "Use of square brackets not permitted." );
                        return;
                    }
                    if ( label.indexOf( "]" ) > 0 ) {
                        errorWithMarkup( "Use of square brackets not permitted." );
                        return;
                    }
                    ok();
                }
            }.process();
        }
    }
}
