package hudson.plugins.si;

import hudson.Util;
import hudson.model.AbstractBuild;
import hudson.model.TaskListener;
import hudson.util.ArgumentListBuilder;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.text.SimpleDateFormat;
import java.util.Date;
import org.apache.commons.lang.StringUtils;

import org.apache.commons.lang.time.FastDateFormat;

/*
 * @author Michael Rack
 */
public class SourceIntegrityRlogCommand
extends AbstractSourceIntegrityCommand 
{
    private static final String nl = System.getProperty( "line.separator" );
    protected static final String SEPARATOR_TOKEN = " || ";
    protected static final String SEPARATOR_TOKEN_SEARCH = " \\|\\| ";
    public static final String MKS_DATE_FORMAT = "MM/dd/yyyy kk:mm:ss";
    public static final String MKS_DATE_FORMAT_SEARCH = "MMM dd, yyyy hh:mm:ss a";
    protected final String _configurationPath;
    protected final String _sandbox;
    protected final Date _since;

    
    /**
     * Constructor
     */
    public SourceIntegrityRlogCommand(final String server, final int port, final String username
		, final String password, final String configurationPath, final String sandbox, final Date since)
    {
        super("rlog", server, port, username, password);
        _since = since;
        _configurationPath = configurationPath;
        _sandbox = sandbox;
    }

    /**
     * Parse the log file output
     */
    public static SourceIntegrityChangeLogSet parse(AbstractBuild<?, ?> build, String rlogOutput, TaskListener listener)
    {
        // TODO: If there's an error parsing the log, that should be re-thrown so the caller doesn't
        // write out an invalid file
        
        SimpleDateFormat sdf = new SimpleDateFormat( MKS_DATE_FORMAT_SEARCH );
        SourceIntegrityChangeLogSet changeLogSet = new SourceIntegrityChangeLogSet(build);
        String[] lines = rlogOutput.split("\n");
        for (String line : lines)
        {
            line = line.trim();
            if (line.startsWith("Connecting to") || line.trim().length() <= 0) //skip these messages
            {
                continue;
            }
            final String[] tokens = line.split( SEPARATOR_TOKEN_SEARCH );
            switch (tokens.length)
            {
                // TODO: Fix this so we pull the add/remove/edit field from the log file
                
                case 1: // assuming there is a multi-line comment
                    try
                    {
                        // append to the last entry
                        SourceIntegrityChangeLogEntry entry = changeLogSet.getLast();
                        entry.setMsg(entry.getMsg() + "\n" + tokens[0]);
                        entry.setAction("Add");
                    } 
                    catch (Exception x)
                    {
                        listener.getLogger().println("Unable to parse MKS modification comments!");
                        listener.getLogger().println(x);
                    }
                    break;
                case 7:
                    try
                    {
                        String member = tokens[0];
                        String revision = tokens[1];
                        String author = tokens[2];
                        String date = tokens[3];
                        String changePackageId = tokens[4];
                        String changePackageDesc = tokens[5];
                        String comment = tokens[6];

                        SourceIntegrityChangeLogEntry entry = new SourceIntegrityChangeLogEntry(changeLogSet);
                        entry.setMember(member);
                        entry.setRevision(revision);
                        entry.setMksuser(author);
                        entry.setDate(sdf.parse(date));
                        entry.setCpid(changePackageId);
                        entry.setCpDesc( StringUtils.removeStart( changePackageDesc, "CPSummary:" ) );
                        entry.setMsg( StringUtils.removeStart( comment, "Desc:" ) );
                        entry.setAction("Edit");
                        
                        changeLogSet.add(entry);
                    } 
                    catch (Exception x)
                    {
                        listener.getLogger().println("Unable to parse MKS modification results!");
                        listener.getLogger().println(x);
                    }
                    break;
                default:
                    listener.getLogger().println("Unable to parse MKS rlog results!");
                    break;
            }
        }
        return changeLogSet;
    }

    /**
     * Write a list of files which appear to have changed since the previous build
     * Note: you can call this method multiple times.  Each subsequent call will insert
     * the new changeset into the appropriate location within the file.
     * @param changelogFile
     * @param changeSet
     * @throws java.io.IOException
     */
    public static void writeChangeLog( File changelogFile,
            SourceIntegrityChangeLogSet changeSet, String configPath, String sandboxDir )
    throws IOException
    {
        RandomAccessFile writer = new RandomAccessFile( changelogFile.getAbsolutePath(), "rw" );

        // If we've call this method for the second or more time, we'll seek to 
        // the position to start writing at
        long lastLinePos = -1;
        if ( changelogFile.exists() )
        {
            String readLine = "";
            while ((readLine = writer.readLine()) != null)
            {
                // Found the tag to overwrite
                if (readLine.startsWith( "</changelog>" ))
                {
                    // Go back to previous line, and break out
                    writer.seek( lastLinePos );
                    break;
                }

                // Save position of current line's end
                lastLinePos = writer.getFilePointer();
            }
        }
        // We only write these lines the first time this method is called
        if (lastLinePos <= 0) 
        {
            writer.writeBytes("<?xml version=\"1.0\" encoding=\"UTF-8\"?>" + nl);
            writer.writeBytes("<changelog>" + nl);
        }

        // TODO: need to display this information in the jelly file

        // Include information about the project which these changes belong to
        writer.writeBytes( String.format(
                "\t<sandboxChanges name=\"%s\" configPath=\"%s\">)" + nl
                , Util.xmlEscape( sandboxDir ), Util.xmlEscape( configPath ) ) );

        // Print out every entry in the set.
        for (SourceIntegrityChangeLogEntry changeEntry : changeSet)
        {
            writer.writeBytes("\t\t<modification>" + nl);
            writer.writeBytes(String.format("\t\t\t<action>%s</action>" + nl,
                    Util.xmlEscape(changeEntry.getAction())));
            writer.writeBytes(String.format("\t\t\t<member>%s</member>" + nl, 
                    Util.xmlEscape(sandboxDir + "/" + changeEntry.getMember().replace("\\","/"))));
            writer.writeBytes(String.format("\t\t\t<revision>%s</revision>" + nl,
                    Util.xmlEscape(changeEntry.getRevision())));
            writer.writeBytes(String.format("\t\t\t<date>%s</date>" + nl,
                    FastDateFormat.getInstance(MKS_DATE_FORMAT).format(changeEntry.getDate())));
            writer.writeBytes(String.format("\t\t\t<cpid>%s</cpid>" + nl,
                    Util.xmlEscape(changeEntry.getCpid())));
            writer.writeBytes(String.format("\t\t\t<mksuser>%s</mksuser>" + nl,
                    Util.xmlEscape(changeEntry.getMksuser())));
            writer.writeBytes(String.format("\t\t\t<msg>%s</msg>" + nl,
                    Util.xmlEscape(changeEntry.getMsg())));
            writer.writeBytes(String.format("\t\t\t<cpDesc>%s</cpDesc>" + nl,
                    Util.xmlEscape(changeEntry.getCpDesc())));
            writer.writeBytes("\t\t</modification>" + nl);
        }
        writer.writeBytes("\t</sandboxChanges>" + nl);
        writer.writeBytes("</changelog>" + nl);
        writer.close();
    }
    

    /**
     * Generate an rlog command to see if there are changes
     */
    @Override
    public ArgumentListBuilder getCommand()
    {
        // Check to see if the sandbox is different than the head of the project
        /*ArgumentListBuilder cmd = super.getCommand();
        cmd.add("--sandbox=" + _sandbox + File.separator + "project.pj");
        cmd.add("--recurse");
        cmd.add("--noHeaderFormat");
        cmd.add("--noTrailerFormat");
        cmd.add("--format={membername}" + SEPARATOR_TOKEN
                + "{revision}" + SEPARATOR_TOKEN
                + "{author}" + SEPARATOR_TOKEN
                + "{date}" + SEPARATOR_TOKEN
                + "{cpid}" + SEPARATOR_TOKEN
                + "{description}"
		+ "\n"
		);
        cmd.add("--filter=changed:missing,changed:newmem,changed:sync");
        cmd.add("--filter=!changed:working");
        cmd.add("--revision=:head");
        */

        // Note: Somewhere, Hudson puts single-quotes around arguments with spaces.
        ArgumentListBuilder cmd = super.getCommand();
        cmd.add("--project=" + _configurationPath);
        cmd.add("--recurse");
        cmd.add("--noHeaderFormat");
        cmd.add("--noTrailerFormat");
        cmd.add("--rfilter=branch::current");
        cmd.add("--rfilter=anyspecial");
        // DO NOT put quotes anywhere this
        cmd.add("--format="
                + "{membername}" + SEPARATOR_TOKEN
                + "{revision}" + SEPARATOR_TOKEN
                + "{author}" + SEPARATOR_TOKEN
                + "{date}" + SEPARATOR_TOKEN
                + "{cpid}" + SEPARATOR_TOKEN
                + "CPSummary:{cpsummary}" + SEPARATOR_TOKEN
                + "Desc:{description}"
		+ "\n"
		);
        // We must put single quotes around dates, unless the date is empty
        String beginDate = "'" + FastDateFormat.getInstance(MKS_DATE_FORMAT_SEARCH).format(_since) + "'";
        String endDate = "";
        cmd.add("--rfilter=daterange:" + beginDate + "-" + endDate + "");
        //cmd.add("--rfilter=daterange:'Sep 11, 2001 08:46:15 AM'-");

        return cmd;
    }


    /**
     * @param rlogOutput Log file to parse for changes
     * @return If changes have been detected in the project since the previous build
     */
    public static boolean changesDetected( String rlogOutput )
    {
        if (null == rlogOutput || 0 == rlogOutput.length())
        {
            return false;
        }
		
        final String[] lines = rlogOutput.split("\n");
        for (String line : lines)
        {
            if (line.startsWith("Connecting to") || line.trim().length() <= 0) //skip these messages
            {
                continue;
            }
            // Found a line that doesn't start with "connecting to"; assume that means a change
            return true;
        }

        // We think there were no changes
        return false;
    }
    
}
