/*
 * Copyright (C) 2007-2008  Alessandro Melandri
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * */

package net.melandri.jtextfileparser.beans;

import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.Iterator;

/**
 * @author Alessandro Melandri
 * @version 1.0
 * */
public class TextFile extends TextFileManager{

	private String fileName;
	private ArrayList rows;
	
	
	/**
	 * This constructor will return a TextFile object for the specified text file.
	 * 
	 * @param filename  Name of the file
	 * @param filePath  Path where the file is stored.
	 * @param sep 		Separator beetween fields.
	 * */
	public TextFile(String fileName, String filePath, String sep) throws FileNotFoundException{
		super(fileName, filePath, sep);
		this.fileName = fileName;
		this.rows = read();
	}
	
	
	
	
	
	/**
	 * This method will return the original row
	 * 
	 * @param ronNumber number of the row in the file
	 * 
	 * @return Returns the original row
	 * */
	public String getStringRow(int rowNumber){
		return ((Row)rows.get(rowNumber)).getSimpleRow();
	}
	
	
	
	
	/**
	 * This method will return a Row object corresponding to the specified index
	 * 
	 * @param ronNumber number of the row in the file
	 * 
	 * @return Returns the corresponding Row object
	 * */
	public Row getRow(int rowNumber){
		return (Row)rows.get(rowNumber);
	}
	
	
	
	/**
	 * @return Returns an ArrayList of Row objects
	 * */
	public ArrayList getRows() {
		return rows;
	}
	
	
	
	/**
	 * This method will return the number of rows in the file
	 *  
	 * @return Returns the number of rows in the file
	 * */
	public int lenght(){
		return rows.size();
	}
	
	
	
	/**
	 * This method checks if all the rows have the same number of fields
	 * 
	 * @return  Returns true if all the the rows have the same number of fields, otherwise it returns false
	 * */
	public boolean checkRowsLength(){
		
		boolean check = true;
		
		Iterator it = rows.iterator();
		int sizeHelper = 0;
		
		while (it.hasNext() && check){
			
			Row thisRow = (Row)it.next();
			int size = thisRow.getFields().length;
			
			if (sizeHelper == 0)
				sizeHelper = size;
			else if (size != sizeHelper)
				check = false;
		}
		
		return check;
	}

	
	
	/**
	 * This method checks if all the rows have the specified number of fields
	 * 
	 * @param fieldsNumber The number of expected fields in a row.
	 * 
	 * @return  Returns an ArrayList of rows that don't have the specified number of fields
	 * */
	public ArrayList checkRowsLength(int fieldsNumber){
		
		ArrayList wrongRows = new ArrayList();
		
		Iterator it = rows.iterator();

		while (it.hasNext()){
			
			Row thisRow = (Row)it.next();
			
			if (thisRow.getFields().length != fieldsNumber)
				wrongRows.add(thisRow);
		}
		
		return wrongRows;
	}

	
	
	/**
	 * This method checks if the specified field has the same length in every row. 
	 * 
	 * @param 	fieldPosition The field to check (numeration starts from 0)
	 * @param 	fieldLength 	Expected field length
	 * 
	 * @return 	Returns an ArraList of Rows.
	 * */
	public ArrayList checkField(int fieldPosition, int fieldLength){
		
		ArrayList wrongRows = new ArrayList();
		
		Iterator it = rows.iterator();
		
		while (it.hasNext()){
			
			Row thisRow = (Row)it.next();
			String field = (String)thisRow.getFields()[fieldPosition];
			
			if (field.length() != fieldLength)
				wrongRows.add(thisRow);
		}
		
		return wrongRows;
	}
	
	
	
	/**
	 * Creates a copy of the text file
	 * 
	 * @param fileName Name of the file
	 * @param filePath Path of the file
	 * */
	public void write(String fileName, String filePath){
		write(this.rows, fileName, filePath);
	}
	
	
	
	/**
	 * Creates a copy of the text file
	 * 
	 * @param fileName Name of the file
	 * @param filePath Path of the file
	 * @param sep		Fields separator
	 * */
	public void write(String fileName, String filePath, String sep){
		write(this.rows, fileName, filePath, sep);
	}
	
	
	
	/**
	 * Creates a new text file
	 * 
	 * @param fileName Name of the file
	 * @param filePath Path of the file
	 * @param rowindex Array of integer containing the indexes of the rows to be exported. Rows indexes start from zero.
	 * */
	public void write(String fileName, String filePath, int[] rowindex){
		
		ArrayList rowsSubset = new ArrayList(rowindex.length);
		
		for (int i = 0 ; i < rowindex.length ; i++){
			rowsSubset.add(this.rows.get(rowindex[i]));
		}
		
		write(rowsSubset, fileName, filePath);
	}
	
	
	
	/**
	 * Creates a new text file
	 * 
	 * @param fileName Name of the file
	 * @param filePath Path of the file
	 * @param rowindex Array of integer containing the indexes of the rows to be exported. Rows indexes start from zero.
	 * @param sep		Fields separator
	 * */
	public void write(String fileName, String filePath, int[] rowindex, String sep){
		
		ArrayList rowsSubset = new ArrayList(rowindex.length);
		
		for (int i = 0 ; i < rowindex.length ; i++){
			rowsSubset.add(this.rows.get(rowindex[i]));
		}
		
		write(rowsSubset, fileName, filePath, sep);
	}
	
	
	
	/**
	 * Creates a new text file
	 * @
	 * @param fileName 		Name of the file
	 * @param filePath 		Path of the file
	 * @param rowindex 		Array of integer containing the indexes of the rows to be exported. Rows indexes start from zero.
	 * @param fieldindex	Array of integer containing the indexes of the fields to be exportes. Fields indexes start from zero.
	 * */
	public void write(String fileName, String filePath, int[] rowindex, int[]fieldindex){
		
		ArrayList rowsSubset = new ArrayList(rowindex.length);
		
		for (int i = 0 ; i < rowindex.length ; i++){
			Row thisRow = (Row)this.rows.get(rowindex[i]);
			
			String[] fields = thisRow.getFields(fieldindex);
			
			Row newRow = new Row(thisRow.getSEP());
			newRow.setFields(fields);
			
			rowsSubset.add(newRow);
		}
		
		write(rowsSubset, fileName, filePath);
	}
	
	
	
	/**
	 * Creates a new text file
	 * @
	 * @param fileName 		Name of the file
	 * @param filePath 		Path of the file
	 * @param rowindex 		Array of integer containing the indexes of the rows to be exported. Rows indexes start from zero.
	 * @param fieldindex	Array of integer containing the indexes of the fields to be exportes. Fields indexes start from zero.
	 * @param sep			Fields separator
	 * */
	public void write(String fileName, String filePath, int[] rowindex, int[]fieldindex, String sep){
		
		ArrayList rowsSubset = new ArrayList(rowindex.length);
		
		for (int i = 0 ; i < rowindex.length ; i++){
			Row thisRow = (Row)this.rows.get(rowindex[i]);
			
			String[] fields = thisRow.getFields(fieldindex);
			
			Row newRow = new Row(thisRow.getSEP());
			newRow.setFields(fields);
			
			rowsSubset.add(newRow);
		}
		
		write(rowsSubset, fileName, filePath, sep);
	}
	
	
	
	/**
	 * @return Returns the name of the file
	 * */
	public String getFileName() {
		return fileName;
	}
	
}
