/*
 * Copyright (C) 2007-2008  Alessandro Melandri
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * */

package net.melandri.jtextfileparser.beans;

import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.Iterator;

import net.melandri.jtextfileparser.utils.JTextFileParserException;

/**
 * @author Alessandro Melandri
 * @version 1.2
 * */
public class TextFile extends TextFileManager{

	private String fileName;
	private ArrayList rows;
	private Row headerRow;
	private boolean hasHeader = false;
	
	
	/**
	 * This constructor will return a TextFile object for the specified text file.
	 * 
	 * @param filename  Name of the file
	 * @param filePath  Path where the file is stored.
	 * @param sep 		Separator beetween fields.
	 * */
	public TextFile(String fileName, String filePath, String sep) throws FileNotFoundException{
		super(fileName, filePath, sep);
		this.fileName = fileName;
		this.rows = read(1);
	}
	
	
	
	/**
	 * This constructor will return a TextFile object for the specified text file.
	 * 
	 * @param filename   Name of the file
	 * @param filePath   Path where the file is stored.
	 * @param sep 		  Separator beetween fields.
	 * @param hasHeader  Switch to specify if the file has a header line
	 * */
	public TextFile(String fileName, String filePath, String sep, boolean hasHeader) throws FileNotFoundException{
		super(fileName, filePath, sep);
		this.hasHeader = hasHeader;
		this.fileName = fileName;
		this.rows = read(0);
		
		if (hasHeader){
			this.headerRow = (Row)this.rows.get(0);
			this.rows.remove(0);
		}
	}
	
	
	
	/**
	 * This method will return the original row
	 * 
	 * @param ronNumber number of the row in the file
	 * 
	 * @return Returns the original row
	 * */
	public String getStringRow(int rowNumber) throws JTextFileParserException{
		
		Row thisRow = null;
		
		try {
			thisRow = (Row)rows.get(rowNumber);
		} catch (IndexOutOfBoundsException e) {
			e.printStackTrace();
			throw new JTextFileParserException("You've specifed a row ID greater than the total row number");
		}

		
		return thisRow.getStringRow();
	}
	
	
	
	/**
	 * This method will return a Row object corresponding to the specified index
	 * 
	 * @param ronNumber number of the row in the file
	 * 
	 * @return Returns the corresponding Row object
	 * */
	public Row getRow(int rowNumber){
		return (Row)rows.get(rowNumber);
	}
	
	
	
	/**
	 * @return Returns an ArrayList of Row objects
	 * */
	public ArrayList getRows() {
		return rows;
	}
	
	
	
	/**
	 * Replaces all rows int he text file.
	 * 
	 * @param rows ArrayList of rows objects
	 * */
	public void replaceRows(ArrayList rows){
		this.rows = rows;
	}
	
	
	
	/**
	 * This method will return the number of rows in the file
	 *  
	 * @return Returns the number of rows in the file
	 * */
	public int lenght(){
		return rows.size();
	}
	
	
	
	/**
	 * This method checks if all the rows have the same number of fields
	 * 
	 * @return  Returns true if all the the rows have the same number of fields, otherwise it returns false
	 * */
	public boolean checkRowsLength(){
		
		boolean check = true;
		
		Iterator it = rows.iterator();
		int sizeHelper = 0;
		
		while (it.hasNext() && check){
			
			Row thisRow = (Row)it.next();
			int size = thisRow.getFields().length;
			
			if (sizeHelper == 0)
				sizeHelper = size;
			else if (size != sizeHelper)
				check = false;
		}
		
		return check;
	}

	
	
	/**
	 * This method checks if all the rows have the specified number of fields
	 * 
	 * @param fieldsNumber The number of expected fields in a row.
	 * 
	 * @return  Returns an ArrayList of rows that don't have the specified number of fields
	 * */
	public ArrayList checkRowsLength(int fieldsNumber){
		
		ArrayList wrongRows = new ArrayList();
		
		Iterator it = rows.iterator();

		while (it.hasNext()){
			
			Row thisRow = (Row)it.next();
			
			if (thisRow.getFields().length != fieldsNumber)
				wrongRows.add(thisRow);
		}
		
		return wrongRows;
	}

	
	
	/**
	 * This method checks if the specified field has the same length in every row. 
	 * 
	 * @param 	fieldPosition The field to check (numeration starts from 0)
	 * @param 	fieldLength 	Expected field length
	 * 
	 * @return 	Returns an ArraList of Rows.
	 * */
	public ArrayList checkField(int fieldPosition, int fieldLength){
		
		ArrayList wrongRows = new ArrayList();
		
		Iterator it = rows.iterator();
		
		while (it.hasNext()){
			
			Row thisRow = (Row)it.next();
			String field = (String)thisRow.getFields()[fieldPosition];
			
			if (field.length() != fieldLength)
				wrongRows.add(thisRow);
		}
		
		return wrongRows;
	}
	
	
	
	/**
	 * Creates a copy of the text file
	 * 
	 * @param fileName Name of the file
	 * @param filePath Path of the file
	 * */
	public void write(String fileName, String filePath){
		write(this.rows, fileName, filePath);
	}
	
	
	
	/**
	 * Creates a copy of the text file
	 * 
	 * @param fileName Name of the file
	 * @param filePath Path of the file
	 * @param sep		Fields separator
	 * */
	public void write(String fileName, String filePath, String sep){
		write(this.rows, fileName, filePath, sep);
	}
	
	
	
	/**
	 * Creates a new text file
	 * 
	 * @param fileName Name of the file
	 * @param filePath Path of the file
	 * @param rowindex Array of integer containing the indexes of the rows to be exported. Rows indexes start from zero.
	 * */
	public void write(String fileName, String filePath, int[] rowindex){
		
		write(getSubset(rowindex,null), fileName, filePath);
	}
	
	
	
	/**
	 * Creates a new text file
	 * 
	 * @param fileName Name of the file
	 * @param filePath Path of the file
	 * @param rowindex Array of integer containing the indexes of the rows to be exported. Rows indexes start from zero.
	 * @param sep		Fields separator
	 * */
	public void write(String fileName, String filePath, int[] rowindex, String sep){
		
		write(getSubset(rowindex,null), fileName, filePath, sep);
	}
	
	
	
	/**
	 * Creates a new text file
	 * @
	 * @param fileName 		Name of the file
	 * @param filePath 		Path of the file
	 * @param rowindex 		Array of integer containing the indexes of the rows to be exported. Rows indexes start from zero.
	 * @param fieldindex	Array of integer containing the indexes of the fields to be exportes. Fields indexes start from zero.
	 * */
	public void write(String fileName, String filePath, int[] rowindex, int[]fieldindex){
		
		write(getSubset(rowindex,fieldindex), fileName, filePath);
	}
	
	
	
	/**
	 * Creates a new text file
	 * @
	 * @param fileName 		Name of the file
	 * @param filePath 		Path of the file
	 * @param rowindex 		Array of integer containing the indexes of the rows to be exported. Rows indexes start from zero.
	 * @param fieldindex	Array of integer containing the indexes of the fields to be exportes. Fields indexes start from zero.
	 * @param sep			Fields separator
	 * */
	public void write(String fileName, String filePath, int[] rowindex, int[]fieldindex, String sep){
		
		write(getSubset(rowindex,fieldindex), fileName, filePath, sep);
	}
	
	
	
	/**
	 * Export the text file to an HTML file
	 * 
	 * @param rows		Rows to be included in the new file
	 * @param fileName 	The name of the file to be created
	 * @param filePath 	The path where the file will be created
	 * @param title		Title of the html page
	 * @param before	Text before the data table (text or HTML)
	 * @param after		Text after the data table (text or HTML)
	 * */
	public void exportHTML(String fileName, String filePath, String title, String before, String after){
		writeHTML(this.rows, fileName, filePath, title, before, after);
	}
	
	
	
	/**
	 * Export the text file to an HTML file
	 * 
	 * @param rows		Rows to be included in the new file
	 * @param fileName 	The name of the file to be created
	 * @param filePath 	The path where the file will be created
	 * @param title		Title of the html page
	 * @param before	Text before the data table (text or HTML)
	 * @param after		Text after the data table (text or HTML)
	 * @param rowindex 		Array of integer containing the indexes of the rows to be exported. Rows indexes start from zero.
	 * */
	public void exportHTML(String fileName, String filePath, String title, String before, String after, int[] rowindex){
		
		writeHTML(getSubset(rowindex,null), fileName, filePath, title, before, after);
	}
	
	
	
	/**
	 * Export the text file to an HTML file
	 * 
	 * @param fileName 	The name of the file to be created
	 * @param filePath 	The path where the file will be created
	 * @param title		Title of the html page
	 * @param before	Text before the data table (text or HTML)
	 * @param after		Text after the data table (text or HTML)
	 * @param rowindex 		Array of integer containing the indexes of the rows to be exported. Rows indexes start from zero.
	 * @param fieldindex	Array of integer containing the indexes of the fields to be exportes. Fields indexes start from zero.
	 * */
	public void exportHTML(String fileName, String filePath, String title, String before, String after, int[] rowindex, int[]fieldindex){
		
		writeHTML(getSubset(rowindex,fieldindex), fileName, filePath, title, before, after);
	}
	
	
	
	/**
	 * @return The name of the file
	 * */
	public String getFileName() {
		return fileName;
	}
	
	
	
	/**
	 * @return Returns the header row
	 * */
	public Row getHeaders(){
		return headerRow;
	}
	
	
	
	/**
	 * @param 	rowindex	Indexes of Rows to be included
	 * @param 	fieldindex Indexes of fields to be included in every row
	 * @return An ArrayList of Row objects.
	 * */
	private ArrayList getSubset(int[] rowindex, int[]fieldindex){
		
		ArrayList rowsSubset = new ArrayList(rowindex.length);
		
		if (fieldindex != null){
			
			for (int i = 0 ; i < rowindex.length ; i++){
				Row thisRow = (Row)this.rows.get(rowindex[i]);
				
				String[] fields = thisRow.getFields(fieldindex);
				
				Row newRow = new Row(thisRow.getSEP());
				newRow.setFields(fields);
				
				rowsSubset.add(newRow);
			}
			
		} else {
			
			for (int i = 0 ; i < rowindex.length ; i++){
				rowsSubset.add(this.rows.get(rowindex[i]));
			}
			
		}
		
		return rowsSubset;
	}
	
	
}
