/*
 * Copyright (C) 2007-2008 Alessandro Melandri
 * 
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package net.melandri.jtextfileparser.utils;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;

import net.melandri.jtextfileparser.beans.Row;
import net.melandri.jtextfileparser.dao.FileManager;
import net.melandri.jtextfileparser.filters.HTMLFileFilter;


/**
 * Html report generator. Used by a TextFile object to build an HTML report
 * 
 * @author Alessandro Melandri
 * 
 * @since 1.3
 * */
public class HTMLBuilder {
	
	/**
	 * Filemanager instance used for writing to filesystem
	 * */
	private FileManager htmlFile = null;
	
	/**
	 * Keeps a table count to generate a unique id for every table
	 * */
	private int tableCount = 0;
	
	/**
	 * Keeps a sections count to generate a unique id for every section
	 * */
	private int sectionsCount = 0;
	
	/**
	 * HTML file filter
	 * */
	private HTMLFileFilter filter;
	
	
	/**
	 * Deafult css properties
	 * */
	private String[] styles  = {
								"body{font-family: arial, verdana, helvetica; font-size: 12px; padding: 10px; margin: 0;}",
								".jtable{width: 100%; border: 1px solid #e1e1e1;border-collapse:collapse}",
								"tr.even{background: #f1f1f1;}",
								"td.jtd{padding: 3px;border: 1px solid #e1e1e1;}",
								"td.num{width: 40px;text-align:right;font-weight: bold;background: #638c9c;color: #FFFFFF;}"
								};
	
	
	/**
	 * Creates the specified file.
	 * 
	 * @param filter HTMLFileFilter containig informations about the file to be created
	 * */
	public HTMLBuilder(HTMLFileFilter filter){
		this.filter = filter;
		htmlFile = new FileManager(filter.getFileName(),filter.getFilePath());
		htmlFile.createFile();
	}
	
	
	
	/**
	 * Adds doctype, headers and body start tag
	 * <br><br>
	 * If the cssPath have not been defined, the default style sheet will be used
	 * */
	public void addHeaders() {
		
		String headers = "";
		
		try {
			headers += "<!DOCTYPE html\n";
			headers += "     PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\"\n";
			headers += "    \"DTD/xhtml1-transitional.dtd\">\n";

			headers += "<html>\n<head>\n";
			
			if (filter.getPageTitle() != null)
				headers += "\t<title>" + filter.getPageTitle() + "</title>\n";
			
			if (filter.getCssFilePath() != null)
				headers += "\t<link rel=\"Stylesheet\" type=\"text/css\" href=\"" + filter.getCssFilePath() + "\"/>\n";
			else{
				headers += "<style type=\"text/css\">\n";
				
				for (int i = 0; i < styles.length; i++) {
					headers += "\t" + styles[i] + "\n";
				}
				
				headers += "</style>";
			}
			
			headers += "</head>\n<body>";
			
			htmlFile.writeLine(headers);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
	
	
	
	/**
	 * Inserts a table into the page
	 * 
	 * @param rows 		Arraylist of Rows objects
	 * */
	public void addTable(ArrayList rows){
		
		try {
		
			htmlFile.writeLine("<table id=\"table_" + tableCount + "\" cellspacing=\"0\" cellpadding=\"0\" class=\"jtable\">");
			
			Iterator it = rows.iterator();
			
			int rowNumber = 0;
			
			while (it.hasNext()) {
				
				Row row = (Row) it.next();
				
				String[] fields = null;
				
				if (filter.getFields() != null && filter.getFields().length > 0)
					fields = row.getFields(filter.getFields());
				else
					fields = row.getFields();
				
				int passedRowNumber = rowNumber;
				
				if (filter.getPrintFileRowNumbers())
					passedRowNumber = row.getNumber();

				htmlFile.writeLine("\t" + this.addTableRow(fields,passedRowNumber, rowNumber));
				
				rowNumber++;
			}

			htmlFile.writeLine("</table>");
			
			tableCount++;
			
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
	
	
	
	/**
	 * Adds a text section into the page
	 * 
	 * @param text Text to be displayed into the page
	 * */
	public void addSection(String text){
		
		try {
			
			if (text.indexOf("\n") == -1){
				
				htmlFile.writeLine("<div id=\"section" + sectionsCount + "\">\n\t<p>" + text + "</p>\n</div>");
				
			} else {
				
				String[] paragraphs = text.split("\n");
				
				htmlFile.writeLine("<div id=\"section" + sectionsCount + "\">");
				
				for (int i = 0; i < paragraphs.length; i++) {
					htmlFile.writeLine("\t<p>" + paragraphs[i] + "</p>");
				}
				
				htmlFile.writeLine("</div>");
				
			}
			
			sectionsCount++;
			
		} catch (IOException e) {
			e.printStackTrace();
		}
		
	}
	
	
	/**
	 * Adds an header row to a table
	 * *//*
	private String addTableHeader(String[] data){
		
		String tableHeader = "<tr id=\"header\">\n";
		
		for (int i = 0; i < data.length; i++) {
			tableHeader += "\t\t<th id=\"th_c" + i +"\">" + data[i] + "</th>\n";	
		}
		
		tableHeader += "\t</tr>";
		
		return tableHeader;
	}*/
	
	
	
	/**
	 * Adds a row to a table
	 * 
	 * @param data
	 * @param passedRowNumber
	 * @param rowNumber
	 * */
	private String addTableRow(String[] data, int passedRowNumber,int rowNumber){
		
		String type = "even";
		
		if (rowNumber % 2 == 0)
			type = "odd";
		
		String tableRow = "<tr id=\"r"+ passedRowNumber + "\" class=\"" + type + "\">\n";
		
		if (filter.getPrintRowNumbers())
			tableRow += "<td class=\"jtd num\">"+ passedRowNumber +"</td>";
		
		for (int i = 0; i < data.length; i++) {
			tableRow += "\t\t<td id=\"r" + passedRowNumber + "_c" + i +"\" class=\"jtd r" + passedRowNumber + " c" + i +"\">" + data[i] + "</td>\n";	
		}
		
		tableRow += "\t</tr>";
		
		return tableRow;
	}
	
	
	
	/**
	 * Closes body and html tags and saves the file
	 * */
	public void close(){
		
		try {
			
			htmlFile.writeLine("</body>\n</html>");
			htmlFile.closeWrite();
			
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
}