/*
 * Copyright (C) 2007-2008 Alessandro Melandri
 * 
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package net.melandri.jtextfileparser.beans;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;

import net.melandri.jtextfileparser.dao.FileManager;
import net.melandri.jtextfileparser.utils.HTMLBuilder;

/**
 * Class for reading and writing text files
 * 
 * @author Alessandro Melandri
 * @version 1.3
 */
public class TextFileManager {

	String SEP;

	private String filePath;
	private String fileName;
	
	private FileManager fileManager;

	

	/**
	 * This constructor is used when building a TextFile from a collection of data
	 * 
	 * @since 1.3
	 */
	protected TextFileManager(String sep) {
		this.SEP = sep;
	}

	

	/**
	 * Loads the text file from the file system
	 * 
	 * @param fileName Text file name
	 * @param filePath Text file path
	 * @param sep Fields separator
	 * 
	 * @throws FileNotFoundException if the text file is not found
	 */
	protected TextFileManager(String fileName, String filePath, String sep) throws FileNotFoundException {

		this.fileName = fileName;
		this.filePath = filePath;
		SEP = sep;
		
		fileManager = new FileManager(this.fileName, this.filePath);
	}



	/**
	 * Parses the file and returns an ArrayList of Row objects
	 * 
	 * @param numStart   Starting line for parsing
	 * @return 		  Retuns an ArrayList of Row objects
	 */
	protected ArrayList parse(int numStart) {

		ArrayList file = new ArrayList();
		String textRow = null;

		int rowCounter = numStart;

		try {
			
			fileManager.loadFile();

			while ((textRow = fileManager.readLine()) != null) {

				if (!textRow.trim().equals("")) {
					Row thisRow = new Row(textRow, SEP, rowCounter);
					file.add(thisRow);
				}
				rowCounter++;
			}

		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			
			try {
				fileManager.closeRead();
			} catch (IOException e) {
				e.printStackTrace();
			}
			
			
		}


		return file;
	}



	/**
	 * Generates a text file from an ArrayList of Rows
	 * 
	 * @param rows Rows to be included in the new file
	 * @param fileName The name of the file to be created
	 * @param filePath The path where the file will be created
	 * @param rowIndex Ids of Rows to be included into the output file
	 * @param fieldIndex Ids of fields to be included into the output file
	 */
	protected void write(ArrayList rows, String fileName, String filePath, int[] rowIndex, int[] fieldsIndex) {
		write(rows, fileName, filePath, SEP, rowIndex, fieldsIndex);
	}



	/**
	 * Generates a text file from an ArrayList of Rows using the specified fields separator
	 * 
	 * @param rows Rows to be included in the new file
	 * @param fileName The name of the file to be created
	 * @param filePath The path where the file will be created
	 * @param sep Fields separator
	 * @param rowIndex Ids of Rows to be included into the output file
	 * @param fieldIndex Ids of fields to be included into the output file
	 */
	protected void write(ArrayList rows, String fileName, String filePath, String sep, int[] rowIndex, int[] fieldIndex) {

		ArrayList rowsSubset = new ArrayList();
		fileManager.createFile();

		if (rowIndex != null && rowIndex.length > 0) {

			for (int i = 0; i < rowIndex.length; i++) {
				rowsSubset.add(rows.get(rowIndex[i]));
			}

		} else {

			rowsSubset = rows;
		}

		Iterator it = rowsSubset.iterator();

		try {

			while (it.hasNext()) {

				Row row = (Row) it.next();

				String stringRow = "";

				String[] fields = null;

				if (fieldIndex != null && fieldIndex.length > 0)
					fields = row.getFields(fieldIndex);
				else
					fields = row.getFields();

				for (int i = 0; i < fields.length; i++) {

					stringRow += fields[i];

					if (i != fields.length - 1)
						stringRow += sep;
				}

				fileManager.writeLine(stringRow);
			}

		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			try {
				
				fileManager.closeWrite();
				
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}

	
	
	
	/**
	 * Generates an HTML file from an ArrayList of Rows
	 * 
	 * @param rows Rows to be included in the new file
	 * @param fileName The name of the file to be created
	 * @param filePath The path where the file will be created
	 * @param title Title of the html page
	 * @param before Text before the data table (text or HTML)
	 * @param after Text after the data table (text or HTML)
	 */
	protected void writeHTML(ArrayList rows, String fileName, String filePath, String title, String before, String after) {
		writeHTML(rows, fileName, filePath, title, before, after, null, null);
	}
	
	
	


	/**
	 * Generates an HTML file from an ArrayList of Rows
	 * 
	 * @param rows Rows to be included in the new file
	 * @param fileName The name of the file to be created
	 * @param filePath The path where the file will be created
	 * @param title Title of the html page
	 * @param before Text before the data table (text or HTML)
	 * @param after Text after the data table (text or HTML)
	 * 
	 * @since 1.3
	 */
	protected void writeHTML(ArrayList rows, String fileName, String filePath, String title, String before, String after, int[] rowIndex, int[] fieldIndex) {
		
		ArrayList rowsSubset = new ArrayList();

		if (rowIndex != null && rowIndex.length > 0) {

			for (int i = 0; i < rowIndex.length; i++) {
				rowsSubset.add(rows.get(rowIndex[i]));
			}

		} else {
			rowsSubset = rows;
		}
		
		HTMLBuilder htmlBuilder = new HTMLBuilder(fileName, filePath);

		htmlBuilder.addHeaders(title);

		if (before != null && !before.equals(""))
			htmlBuilder.addSection(before);

		htmlBuilder.addTable(rows, fieldIndex);

		if (after != null && !after.equals(""))
			htmlBuilder.addSection(after);

		htmlBuilder.close();

	}

}