/*
 * Copyright (C) 2007-2008 Alessandro Melandri
 * 
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package net.melandri.jtextfileparser.utils;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;

import net.melandri.jtextfileparser.beans.Row;
import net.melandri.jtextfileparser.dao.FileManager;


/**
 * Html report generator. Uses data from a TextFile to build an HTML report
 * 
 * @author Alessandro Melandri
 * 
 * @since 1.3
 * */
public class HTMLBuilder {
	
	/**
	 * Filemanager instance used for writing to filesystem
	 * */
	private FileManager htmlFile = null;
	
	/**
	 * Keeps a table count to generate a unique id for every table
	 * */
	private int tableCount = 0;
	
	/**
	 * Keeps a sections count to generate a unique id for every section
	 * */
	private int sectionsCount = 0;
	
	/**
	 * Path of the style sheet to be applyed to the report
	 * */
	private String cssPath;
	
	
	/**
	 * Deafult css properties
	 * */
	private String[] styles  = {
								"body{font-family: arial, verdana, helvetica; font-size: 12px; padding: 20px; margin: 0;}",
								".jtable{width: 100%; border: 1px solid #e1e1e1;border-collapse:collapse}",
								"tr.even{background: #f1f1f1;}",
								"td.jtd{padding: 3px;border: 1px solid #e1e1e1;}"
								};
	
	
	/**
	 * Creates the specified file. No custom stylesheet is specified, so the report will use deafult css properties
	 * 
	 * @param fileName Export file name
	 * @param filePath Export file path
	 * */
	public HTMLBuilder(String fileName, String filePath){
		htmlFile = new FileManager(fileName,filePath);
		htmlFile.createFile();
	}
	
	
	
	/**
	 * Creates the specified file and sets the custom stylesheet's path
	 * 
	 * @param fileName Export file name
	 * @param filePath Export file path
	 * @param cssPath Path of the css file to be used
	 * */
	public HTMLBuilder(String fileName, String filePath, String cssPath){
		htmlFile = new FileManager(fileName,filePath);
		htmlFile.createFile();
		this.cssPath = cssPath;
		
	}
	
	
	
	/**
	 * Adds doctype, headers and body start tag
	 * <br><br>
	 * If the cssPath have not been specified, it uses the default style sheet
	 * 
	 * @param pageTitle String to be used in the &gt;title&lt; tag
	 * */
	public void addHeaders(String pageTitle) {
		
		String headers = "";
		
		try {
			headers += "<!DOCTYPE html\n";
			headers += "     PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\"\n";
			headers += "    \"DTD/xhtml1-transitional.dtd\">\n";

			headers += "<html>\n<head>\n";
			
			if (pageTitle != null)
				headers += "\t<title>" + pageTitle + "</title>\n";
			
			if (this.cssPath != null)
				headers += "\t<link rel=\"Stylesheet\" type=\"text/css\" href=\"" + cssPath + "\"/>\n";
			else{
				headers += "<style type=\"text/css\">\n";
				
				for (int i = 0; i < styles.length; i++) {
					headers += "\t" + styles[i] + "\n";
				}
				
				headers += "</style>";
			}
			
			headers += "</head>\n<body>";
			
			htmlFile.writeLine(headers);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
	
	
	
	/**
	 * Inserts a table into the page
	 * 
	 * @param rows 		Arraylist of Rows objects
	 * @param fieldIndex	Array of fields to be displayed into the page
	 * */
	public void addTable(ArrayList rows, int[] fieldIndex){
		
		try {
		
			htmlFile.writeLine("<table id=\"table_" + tableCount + "\" cellspacing=\"0\" cellpadding=\"0\" class=\"jtable\">");
			
			Iterator it = rows.iterator();
			
			int rowNumber = 0;
			
			while (it.hasNext()) {
				
				Row row = (Row) it.next();
				
				String[] fields = null;
				
				if (fieldIndex != null && fieldIndex.length > 0)
					fields = row.getFields(fieldIndex);
				else
					fields = row.getFields();

				htmlFile.writeLine("\t" + this.addTableRow(fields,rowNumber));
				
				rowNumber++;
			}

			htmlFile.writeLine("</table>");
			
			tableCount++;
			
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
	
	
	
	/**
	 * Adds a text section into the page
	 * 
	 * @param text Text to be displayed into the page
	 * */
	public void addSection(String text){
		
		try {
			
			if (text.indexOf("\n") == -1){
				
				htmlFile.writeLine("<div id=\"section" + sectionsCount + "\">\n\t<p>" + text + "</p>\n</div>");
				
			} else {
				
				String[] paragraphs = text.split("\n");
				
				htmlFile.writeLine("<div id=\"section" + sectionsCount + "\">");
				
				for (int i = 0; i < paragraphs.length; i++) {
					htmlFile.writeLine("\t<p>" + paragraphs[i] + "</p>");
				}
				
				htmlFile.writeLine("</div>");
				
			}
			
			sectionsCount++;
			
		} catch (IOException e) {
			e.printStackTrace();
		}
		
	}
	
	
	private String addTableHeader(String[] data){
		
		String tableHeader = "<tr id=\"header\">\n";
		
		for (int i = 0; i < data.length; i++) {
			tableHeader += "\t\t<th id=\"th_c" + i +"\">" + data[i] + "</th>\n";	
		}
		
		tableHeader += "\t</tr>";
		
		return tableHeader;
	}
	
	
	
	/**
	 * Adds a row to a table
	 * */
	private String addTableRow(String[] data, int rowNumber){
		
		String type = "even";
		
		if (rowNumber % 2 == 0)
			type = "odd";
		
		String tableRow = "<tr id=\"r"+ rowNumber + "\" class=\"" + type + "\">\n";
		
		for (int i = 0; i < data.length; i++) {
			tableRow += "\t\t<td id=\"r" + rowNumber + "_c" + i +"\" class=\"jtd r" + rowNumber + " c" + i +"\">" + data[i] + "</td>\n";	
		}
		
		tableRow += "\t</tr>";
		
		return tableRow;
	}
	
	
	
	/**
	 * Closes body and html tags and saves the file
	 * */
	public void close(){
		
		try {
			
			htmlFile.writeLine("</body>\n</html>");
			htmlFile.closeWrite();
			
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
	
}
