/*
 * Copyright (C) 2007-2009 Alessandro Melandri
 * 
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package net.melandri.jtextfileparser.dao;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;

import net.melandri.jtextfileparser.utils.StringUtils;

/**
 * Class for accessing files
 * 
 * @author Alessandro Melandri
 */
public class FileManager {
	
	private String fileName;
	private String filePath;
	private String fullPath;
	
	private FileWriter fileWriter;
	private BufferedWriter bufferedWriter;
	private BufferedReader bufferedReader;
	
	
	/**
	 * Constructor: sets file name and path and composes the full path to the file.<br>
	 * Trailing slash is not mandatory.
	 * 
	 * @param fileName Name of the file
	 * @param filePath Path of the file
	 * */
	public FileManager(String fileName, String filePath){
		this.fileName = fileName;
		this.filePath = filePath;
		
		composeFilePath();
	}
	
	
	
	
	/**
	 * Creates a file on the file system and sets up the BufferedWriter
	 * 
	 * @param fileName Name of the file
	 * @param filePath Path of the file
	 */
	public void createFile() {

		try {
			this.fileWriter = new FileWriter(fullPath);
			this.bufferedWriter = new BufferedWriter(fileWriter);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
	
	
	
	
	/**
	 * Loads a text file form the file system
	 * 
	 * @return Returns a BufferedReader for the specified text file
	 * @throws IOException
	 */
	public void loadFile() throws FileNotFoundException {
		bufferedReader = new BufferedReader(new InputStreamReader(new FileInputStream(this.fullPath)));
	}
	

	

	/**
	 * Reads the next line from the file
	 * 
	 * @throws IOException
	 * */
	public String readLine() throws IOException{		
		return bufferedReader.readLine();
	}
	
	
	
	
	/**
	 * Writes a row to the file
	 * 
	 * @param row String to be written to the file
	 * 
	 * @throws IOException
	 * */
	public void write(String row) throws IOException {
		bufferedWriter.write(row);
		bufferedWriter.flush();
	}
	
	
	
	
	/**
	 * Writes a row to the file and goes to new line
	 * 
	 * @param row String to be written to the file
	 * 
	 * @throws IOException
	 * */
	public void writeLine(String row) throws IOException {
		this.write(row + "\n");
	}
	

	
	
	/**
	 * Closes read objects for (BufferedReader).
	 * 
	 * @throws IOException
	 * */
	public void closeRead() throws IOException{
		
		if (bufferedReader != null)
			bufferedReader.close();	
	}
	
	
	
	
	/**
	 * Closes write objects (BufferedWriter and FileWriter).
	 * 
	 * @throws IOException
	 * */
	public void closeWrite() throws IOException{
		
		if (bufferedWriter != null)
			bufferedWriter.close();
		
		if (fileWriter != null)
			fileWriter.close();
	}
	
	
	
	/**
	 * Check if the filepath has the trailing slash and composes the full path of a file
	 */
	private void composeFilePath() {
		this.fullPath = StringUtils.composeFilePath(this.filePath,this.fileName);
	}

}
