package sensorBoard.sensor;

import sensorBoard.ISensorBoardComponent;
import sensorBoard.ISensorBoardComponentListener;

import com.CharNotValidException;
import com.Event;
import com.LedsManager;
import com.ParamNotFoundException;
import com.ParamsLabel;
import com.sun.spot.sensorboard.EDemoBoard;
import com.sun.spot.sensorboard.peripheral.ILightSensor;
import com.sun.spot.sensorboard.peripheral.ILightSensorThresholdListener;
import com.sun.spot.resources.transducers.LEDColor;

/**
 * Provides a listener class for Light Sensor. 
 * Send a backEvent to own manager when the light value changes from the threshold value
 * @author Francesco Aiello, Antonio Guerrieri, Raffaele Gravina
 *
 */
public class LightListener implements ILightSensorThresholdListener, ISensorBoardComponentListener {
	
	private ISensorBoardComponent sensorBoardComponent;
	private String sensorID;
	private	Event backEvent;
	private int lowValue;
	private int highValue;
	private ILightSensor lightSensor;
	boolean permanent;
	
	/**
	 * Creates a new LightListener
	 * @param sensorBoardComponent component that invokes the constructor and manages the event transmission
	 * @param sensorID own ID
	 * @param backEvent to send when the light values changes from the threshold values
	 */
	public LightListener (ISensorBoardComponent sensorBoardComponent, String sensorID, Event backEvent) {

		
		try {
			this.sensorBoardComponent = sensorBoardComponent;
			this.sensorID = sensorID;
			this.backEvent = backEvent;
			this.backEvent.setParam(ParamsLabel.SNS_ID, sensorID);
			
			if (backEvent.getType() == Event.PERMANENT)
				this.permanent = true;
			else
				this.permanent = false;
			this.lightSensor = EDemoBoard.getInstance().getLightSensor();
			this.lightSensor.addILightSensorThresholdListener(this);
			this.lightSensor.enableThresholdEvents(true);
			this.lowValue = Integer.parseInt(backEvent.getParam(ParamsLabel.LGH_THRESHOLD_LOW_VALUE));
			this.highValue = Integer.parseInt(backEvent.getParam(ParamsLabel.LGH_THRESHOLD_HIGH_VALUE));
			this.lightSensor.setThresholds(this.lowValue, this.highValue);
			this.lightSensor.enableThresholdEvents(true);
		}
		catch (NumberFormatException e) {
			LedsManager.error(LEDColor.YELLOW);
			e.printStackTrace();
		}
		catch (ParamNotFoundException e) {
			LedsManager.error(LEDColor.YELLOW);
			e.printStackTrace();
		}
		catch (CharNotValidException e) {
			LedsManager.error(LEDColor.YELLOW);
			e.printStackTrace();
		}
		
		
	}
	

	public void thresholdChanged(ILightSensor lightSensor, int low, int high) {
		try {
			backEvent.setParam(ParamsLabel.LGH_THRESHOLD_LOW_VALUE, "" + low);
			backEvent.setParam(ParamsLabel.LGH_THRESHOLD_HIGH_VALUE, "" + high);
			this.sensorBoardComponent.send(backEvent);
			this.lightSensor.enableThresholdEvents(true);
		}
		catch (CharNotValidException e) {
			LedsManager.error(LEDColor.YELLOW);
			e.printStackTrace();
		}
	}
	
	public void thresholdExceeded(ILightSensor lightSensor, int value) {
		try {
			if (value <=  this.lowValue || value >= this.highValue) {
				backEvent.setParam(ParamsLabel.LGH_LIGHT_VALUE, "" + value);
				this.sensorBoardComponent.send(backEvent);
				if (this.permanent)
					this.lightSensor.enableThresholdEvents(true);
				else 
					this.sensorBoardComponent.terminateListener(this.sensorID);
			}
		}
		catch (CharNotValidException e) {
			LedsManager.error(LEDColor.YELLOW);
			e.printStackTrace();
		}		
	}
	
	public boolean terminateListener () {
		this.lightSensor.removeILightSensorThresholdListener(this);
		return true;
	}

	public String getSensorBoardComponentID() {
		return sensorID;
	}
	
}
