package ch.simas.nbtostring.util;

import org.netbeans.api.annotations.common.CheckForNull;
import org.netbeans.api.annotations.common.NonNull;

/** Class used to represent problems encountered when performing
 * various refactoring calls. Problems can be chained (using setNext method)
 * - every problem can point to the following problem.
 *
 * @author Martin Matula
 */
public class Problem {
    private final boolean fatal;
    private final String message;
    private Problem next = null;

    /** Creates new instance of Problem class.
     * @param fatal Indicates whether the problem is fatal.
     * @param message Textual description of the problem.
     */
    public Problem(boolean fatal, @NonNull String message) {
        this.fatal = fatal;
        this.message = message;
    }

    /** Indicates whether the problem is fatal.
     * @return <code>true</code> if the problem is fatal, otherwise returns <code>false</code>.
     */
    public boolean isFatal() {
        return fatal;
    }

    /** Returns textual description of the problem.
     * @return Textual description of the problem.
     */
    @NonNull
    public String getMessage() {
        return message;
    }

    /** Returns the following problem (or <code>null</code> if there none).
     * @return The following problem.
     */
    @CheckForNull
    public Problem getNext() {
        return next;
    }

    /**
     * Sets the following problem. The problem can be set only once - subsequent
     * attempts to call this method will result in IllegalStateException.
     * @param next The following problem.
     * @throws java.lang.IllegalStateException subsequent attempts to call this method will result in IllegalStateException.
     */
    public void setNext(@NonNull Problem next) throws IllegalStateException {
        if (this.next != null) {
            throw new IllegalStateException("Cannot change \"next\" property of Problem."); //NOI18N
        }
        this.next = next;
    }
}
