/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package com.ohell.rules;

import com.ohell.data.CardBasic;
import com.ohell.data.enums.CardColor;
import com.ohell.data.enums.CardStrength;
import com.ohell.exception.OhellRulesManagerException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * Manager class that presents public static methods implementing ohell game's rules logic.
 * Those methods can be used to determine what card can be played next, who takes the hand etc.
 * @author vasko
 */
public class OhellRulesManager {

    private static final int QUARTER_OF_ALL_CARDS = 13;
    private static final int TRUMP_ROUNDS_COUNT = 12;
    private static final int FIRST_FRIENDLY_ROUNDS_COUNT = 4;
    public static final int PLAYERS_COUNT = 4;
    private static final int GAME_ROUNDS = 16;
    /**
     * List holding the cardsStrengths in Ohell card game ascending - from smalles to biggest - 2, 3, 4... K, A
     */
    private static final List<CardStrength> cardStrengthAscending = Arrays.asList(CardStrength.TWO, CardStrength.THREE, CardStrength.FOUR, CardStrength.FIVE, CardStrength.SIX, CardStrength.SEVEN, CardStrength.EIGHT, CardStrength.NINE, CardStrength.TEN, CardStrength.JACK, CardStrength.QUEEN, CardStrength.KING, CardStrength.ACE);

    /**
     * @param firstCard - the first card played in the hand
     * @param cards - the list of possible cards to choose a move from
     * @return List with indexes, representing indexes of cards in the passed cards List, that can be played on this turn
     */
    public static List<Integer> getCardsThatCanBePlayed(CardBasic firstCard, List<CardBasic> cards) {
        List<Integer> result = null;
        if (cards != null && cards.size() > 0) {
            result = new ArrayList<Integer>();
            //if there are cards with the same color as first one played - should play one of them
            for (int i = 0; i < cards.size(); i++) {
                CardBasic cTemp = cards.get(i);
                if (cTemp.getColor().equals(firstCard.getColor())) {
                    result.add(i);
                }
            }
            //if there are no cards as the first one played - all moves are possible
            if (result.size() == 0) {
                for (int i = 0; i < cards.size(); i++) {
                    result.add(i);
                }
            }
        }
        return result;
    }

    public static int getWinnerHandIndex(int firstPlayerInHand, CardColor trump, List<? extends CardBasic> cards) throws OhellRulesManagerException {
        int result = -1;
        if(trump == null){
            result = getWinnerHandIndex_NoTrump(cards);
        }else{
            result = getWinnerHandIndex_Trump(trump, cards);
        }
        if(result != -1){
            //getWinnerHandIndex method returns the index of winningCard in the cards array
            result = (firstPlayerInHand + result) % PLAYERS_COUNT;
        }
        return result;
    }

    /*this method returns the index of winningCard in the cards array*/
    public static int getWinnerHandIndex_NoTrump(List<? extends CardBasic> cards) throws OhellRulesManagerException {
        if (cards == null || cards.size() != 4) {
            throw new OhellRulesManagerException("The cards given for evaluation are not 4");
        }
        if (cards.get(0) == null) {
            throw new OhellRulesManagerException("Card with index " + 0 + " is null");
        }

        int resultIndex = 0;
        CardBasic winnerCard = cards.get(0);
        for (int i = 1; i < cards.size(); i++) {
            CardBasic temp = cards.get(i);
            if (temp == null) {
                throw new OhellRulesManagerException("Card with index " + i + " is null");
            }

            if (!temp.getColor().equals(winnerCard.getColor())) {
                //if temp is not of the color of winner - it will not win
                continue;
            } else {
                //if temp is of the color of winner - increment if temp is bigger
                if (isFirstBiggerThanSecond(temp, winnerCard)) {
                    winnerCard = temp;
                    resultIndex = i;
                }
            }
        }
        return resultIndex;
    }

    /**
     * This method returns the index of winningCard in the cards array
     *
     * This method should be used to determine the winner of the hand as well as to determine the player
     * that is to play first in the next Hand - in ohell as in most games they coincide
     * @param trump the trump under which the Hand was played
     * @param cards the four cards given by the players in the hand
     * @return the index (zero based) of the winner of the Hand - <b>important<b/> - it is not the seatIndex of a player.
     * The returned index is actually the index of the card in the given card array. The seatIndex is further to be determined
     * by the result returned from this method.
     * @throws com.ohell.exception.OhellRulesManagerException
     */
    public static int getWinnerHandIndex_Trump(CardColor trump, List<? extends CardBasic> cards) throws OhellRulesManagerException {
        if (trump == null) {
            throw new OhellRulesManagerException("Trump is not specified");
        }
        if (cards == null || cards.size() != 4) {
            throw new OhellRulesManagerException("The cards given for evaluation are not 4");
        }
        if (cards.get(0) == null) {
            throw new OhellRulesManagerException("Card with index " + 0 + " is null");
        }
        int resultIndex = 0;
        CardBasic winnerCard = cards.get(0);
        boolean isTrumpPlayed = winnerCard.getColor().equals(trump) ? true : false;
        for (int i = 1; i < cards.size(); i++) {
            CardBasic temp = cards.get(i);
            if (temp == null) {
                throw new OhellRulesManagerException("Card with index " + i + " is null");
            }
            if (isTrumpPlayed) {//trump is already played within previous cards
                if (!temp.getColor().equals(trump)) { //if temp is not a trump - it will not win
                    continue;
                } else {//if temp is trump and is bigger than current winner - increment
                    if (isFirstBiggerThanSecond(temp, winnerCard)) {
                        winnerCard = temp;
                        resultIndex = i;
                    }
                }
            } else {//trump is not played withing previous cards
                if (temp.getColor().equals(trump)) {//if temp is trump - it is the currentWinner
                    winnerCard = temp;
                    resultIndex = i;
                } else {
                    if (!temp.getColor().equals(winnerCard.getColor())) {
                        //if temp is not a trump and is not of the color of winner - it will not win
                        continue;
                    } else {
                        //if temp is of the color of winner - increment if temp is bigger
                        if (isFirstBiggerThanSecond(temp, winnerCard)) {
                            winnerCard = temp;
                            resultIndex = i;
                        }
                    }
                }
            }
            //update if trump is played or not
            isTrumpPlayed = temp.getColor().equals(trump) ? true : isTrumpPlayed;
        }
        return resultIndex;
    }

    public static boolean isFirstBiggerThanSecond(CardBasic first, CardBasic second) {
        if (second == null) {
            return true;
        }
        if (first == null) {
            return false;
        }
        return cardStrengthAscending.indexOf(first.getStrength()) >
                cardStrengthAscending.indexOf(second.getStrength());
    }

    /**     
     * @param announcedHands - the hands announced at the beginning of the round
     * @param takenHands - the hands taken at the end of the round
     * @return the points won by the player in the current round
     */
    public static int getPointsForTurn(int announcedHands, int takenHands) {
        if (announcedHands == takenHands) {
            return announcedHands * announcedHands + 10;
        } else {
            return 0;
        }
    }

    /**
     * When player is last it has boundaries on his announce - his announce in sum with all other announces
     * can not be equal to the total number of cards in the round - which is equal to the round number.
     * @param roundNumber equal to the total number of hands in the round
     * @param seatIndex the seatIndex of the player that is to announce
     * @param firstInRoundToAnnounce the first one that's to announce
     * @param currentSumOfHandsAnnounced the sum of all announces made so far
     * @return list of possible announces of the currentSeatIndex
     */
    public static List<Integer> getPossibleAnnounces(int roundNumber, int seatIndex, int currentSumOfHandsAnnounced) {
        List<Integer> result = new ArrayList<Integer>();
        for (int i = 0; i <= roundNumber; i++) {
            result.add(i);
        }
        if (roundNumber > FIRST_FRIENDLY_ROUNDS_COUNT) {
            int firstInRoundToAnnounce = firstToActInCurrentRound(roundNumber);
            System.out.println("FirstInRoundToAnnounce: "+firstInRoundToAnnounce);
            System.out.println("SeatIndex: "+seatIndex);
            boolean isLastToAnnounce = (firstInRoundToAnnounce + 3) % PLAYERS_COUNT == seatIndex;
            System.out.println("IsLastToAnnounce: "+isLastToAnnounce);
            if (isLastToAnnounce && currentSumOfHandsAnnounced <= roundNumber) {
                int forbidenAnnounce = roundNumber - currentSumOfHandsAnnounced;
                result.remove(forbidenAnnounce);
            }
        }
        return result;
    }

    /**
     * Server and client will hold bean-like data (data that is to be processed by controllers).
     * But actual operations on this data should be performed by game controller - those are all
     * actions that are executed both on the server and the client.This method is very simple but
     * yet descriptive example of logic that is unrelated neither to client nor to server, but to
     * the game controller which in this case is the OhellRulesManager
     * @param currentAnnouncesCount the count of current announces made
     * @return whether or not all players in the current round have made announces
     */
    public static boolean isRoundAnnouncesMade(int currentAnnouncesCount) {
        return currentAnnouncesCount == PLAYERS_COUNT;
    }

    /**
     * @param numbersOfCardsPlayed - the number of the players played so far
     * @return whether or not all the players in the hand has played their card
     */
    public static boolean isCurrentHandOver(int numbersOfCardsPlayed) {
        return numbersOfCardsPlayed == PLAYERS_COUNT;
    }

    /**
     *
     * @param roundNumber - the round number which is equal to the number of cards each player has in the beginning of the round
     * @param handsPlayedSoFar - the number of hands played so far in this round
     * @return whether or not the round is over
     */
    public static boolean isCurrentRoundOver(int roundNumber, int handsPlayedSoFar) {
        if (roundNumber < TRUMP_ROUNDS_COUNT) {
            //roundNumber is NO MORE zeroBased while cards in each round start from 1
            return roundNumber == handsPlayedSoFar;
        } else {
            return QUARTER_OF_ALL_CARDS == handsPlayedSoFar;
        }
    }

    /**
     * The game is over if all expected rounds are played
     * @param roundsPlayedSoFar - the number of rounds played so far in the game
     * @return whether or not the game is over
     */
    public static boolean isGameOver(int roundsPlayedSoFar) {
        return roundsPlayedSoFar == GAME_ROUNDS;
    }

    /**
     * It is assumed that the player with seatIndex = 0 will always begin first on the first round.
     * This gives equal chance for every player to be seatIndex = 0 and be first on first round.
     * No need for another random function that will determine which seatIndex is first and after that
     * one more index in the game logic to be remembered in the implementation.
     * @param roundIndex - IT IS NOT ZERO BASED...
     * @return seatIndex of the player that is to act first in the current round
     */
    public static int firstToActInCurrentRound(int roundIndex) {
        return (roundIndex-1) % PLAYERS_COUNT;
    }

    /**
     * Be careful when using this method - it will give the next player seat index that should act next
     * but will not check whether or not the curremt hand/round is over.
     * @param prevPlayerSeatIndex seatIndex of the last player that participate action
     * @return the seatIndex of the player that is next to act
     */
    public static int nextPlayerToAct(int prevPlayerSeatIndex) {
        return (prevPlayerSeatIndex + 1) % PLAYERS_COUNT;
    }

    public static int prevPlayerToAct(int nextPlayerSeatIndex) {
        int index = nextPlayerSeatIndex - 1;
        index = index < 0 ? index + PLAYERS_COUNT : index;
        return (index) % PLAYERS_COUNT;
    }

    /**
     *
     * @param prevRoundIndex
     * @return the next round index which in the ohell game is previousRoundIndex plus one
     */
    public static int getNextRoundIndex(int prevRoundIndex) {
        return prevRoundIndex + 1;
    }

    public static int getFirstRoundIndex() {
        return 1;
    }    
}
