package com.ohell.data;

import com.ohell.exception.OhellRulesManagerException;
import com.ohell.relations.GamePlayerRel;
import com.ohell.rules.OhellRulesManager;
import com.ohell.util.Util;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.OneToMany;
import javax.persistence.OneToOne;
import javax.persistence.Transient;
import static javax.persistence.GenerationType.IDENTITY;

@Entity
public class OhellGame implements CardGame, Serializable {

    private static final int OHELL_PLAYERS_REQUIRED = 4;
    private static final int OHELL_ROUNDS_COUNT = 16;
    private int currentRoundIndex;
//    private Set<GamePlayerRel> gamePlayerRels = new HashSet<GamePlayerRel>();
    private List<GamePlayerRel> gamePlayerRels = new ArrayList<GamePlayerRel>();
    private List<Round> rounds;
    private ScoreTable score;
    private Dealer dealer;
    private Long id;
    private int seatIndexToPlayNext;
    private int firstDealerSeatIndex;

    public OhellGame() {
    }

    public OhellGame(List<Player> players, Dealer dealer) {
        // Create dealer
        this.dealer = dealer;
        // Shuffle players and create gamePlayerRels
        Util.shuffle(players);
        for (int i = 0; i < players.size(); i++) {
            GamePlayerRel tmpGPRel = new GamePlayerRel(this, players.get(i));
            tmpGPRel.setSeatIndex(i);
            gamePlayerRels.add(tmpGPRel);
        }

        // Create first round
        currentRoundIndex = 0;
        rounds = new ArrayList<Round>();
        Round firstRound = new Round(1, dealer, gamePlayerRels, firstDealerSeatIndex);
        rounds.add(firstRound);

        seatIndexToPlayNext = 0;

        //init game score
        List<PlayerGameScore> pList = new ArrayList<PlayerGameScore>();
        PlayerGameScore temp;
        for (Player player : players) {
            temp = new PlayerGameScore(player);
            pList.add(temp);
        }
        score = new ScoreTable(pList, this);
    }

    public int getFirstDealerSeatIndex() {
        return firstDealerSeatIndex;
    }

    public void setFirstDealerSeatIndex(int firstDealerSeatIndex) {
        this.firstDealerSeatIndex = firstDealerSeatIndex;
    }

    public void setupNextRound() {
    }

    public int getCurrentRoundIndex() {
        return currentRoundIndex;
    }

    public void setCurrentRoundIndex(int val) {
        this.currentRoundIndex = val;
    }

    @Transient
    public Dealer getDealer() {
        return dealer;
    }

    public void setDealer(Dealer val) {
        this.dealer = val;
    }

    @OneToMany(cascade = CascadeType.ALL, fetch = FetchType.LAZY, mappedBy = "game")
    public List<GamePlayerRel> getGamePlayerRels() {
        return gamePlayerRels;
    }

    public void setGamePlayerRels(List<GamePlayerRel> gamePlayerRels) {
        this.gamePlayerRels = gamePlayerRels;
    }

    @OneToMany(cascade = CascadeType.ALL, fetch = FetchType.LAZY)
    public List<Round> getRounds() {
        return rounds;
    }

    public void setRounds(List<Round> val) {
        this.rounds = val;
    }

    @OneToOne
    public ScoreTable getScore() {
        return score;
    }

    public void setScore(ScoreTable val) {
        this.score = val;
    }

    @Id
    @GeneratedValue(strategy = IDENTITY)
    @Column(name = "id", unique = true, nullable = false)
    public Long getId() {
        return id;
    }

    public void setId(Long id) {
        this.id = id;
    }

    @Transient
    public int getSeatIndexToPlayNext() {
        return seatIndexToPlayNext;
    }

    public void incrementSeatIndexToPlayNext(int seatIndexToPlayNext) {
        this.seatIndexToPlayNext = seatIndexToPlayNext;
    }

    @Transient
    public boolean isGameOver() {
        boolean result = (rounds.size() == OHELL_ROUNDS_COUNT) && (rounds.get(OHELL_ROUNDS_COUNT - 1).isRoundOver());
        return result;
    }

    public void updateGameScore(Round r) {
        if (!r.isRoundOver()) {
            System.out.println("Error : Round is not Finished!");
            return;
        } else {
            int a;
            int t;
            int points;
            List<PlayerInRound> pList = r.getPlayers();
            for (PlayerInRound playerInRound : pList) {
                a = playerInRound.getHandsAnnounced().get(0).getCode();
                t = playerInRound.getTakenHands().size();
                PlayerGameScore temp = score.getChosenPlayerGameScore(playerInRound.getPlayer());
                if (temp == null) {
                    System.out.println("Wrong player mapping between game and gameScore");
                } else {
                    temp.addRoundStatistics(r.getRoundNumber(), a, t);
                }
            }
        }
    }

    public void makeMove(Player p, Card c) {
        Round r = rounds.get(rounds.size() - 1);
        List<PlayerInRound> pList = r.getPlayers();
        PlayerInRound temp = getPlayerInRound(pList, p);
        if (temp == null) {
            System.out.println("Wrong Player mapping between gamePlayers and MakeMoveForPlayer object in OgellGame.makeMove()");
        } else {
            if (r.currentHand().isHandComplete()) {
                System.out.println("Error in updating Hand with Player move - Hand is already complete");
            } else {
                GamePlayerRel gpr = getChosenGamePlayerRel(p);
                r.currentHand().getPlayerCardPairs().add(new PlayerCardPair(gpr.getSeatIndex(), c));
                int nextPlayerToAct = 0;
                try {
                    if (r.currentHand().isHandComplete()) {
                        if (r.isRoundOver()) {
                            //TODO : implement logic here for end of round.
//                            initNextRound();
                        } else {
                            int firstPlayerInHand = OhellRulesManager.nextPlayerToAct(r.getIndexOfDealerPlayer());
                            List<Card> cards = getCardsPlayeedInHand(r);
                            nextPlayerToAct = OhellRulesManager.getWinnerHandIndex(firstPlayerInHand, r.getTrump().getColor(), cards);
                            r.initNextHand(nextPlayerToAct);
                        }
                    } else {
                        nextPlayerToAct = OhellRulesManager.nextPlayerToAct(r.getPlayerOnTurn());
                    }
                } catch (OhellRulesManagerException ex) {
                    ex.printStackTrace();
                }
                r.setPlayerOnTurn(nextPlayerToAct);
            }
        }
    }

    @Transient
    private PlayerInRound getPlayerInRound(List<PlayerInRound> pList, Player p) {
        PlayerInRound temp = null;
        for (PlayerInRound playerInRound : pList) {
            if (playerInRound.getPlayer().getId().longValue() == p.getId().longValue()) {
                temp = playerInRound;
                break;
            }
        }
        return temp;
    }

    @Transient
    private List<Card> getCardsPlayeedInHand(Round r) {
        List<Card> cards = new ArrayList<Card>();
        for (PlayerCardPair playerCardPair : r.currentHand().getPlayerCardPairs()) {
            cards.add(playerCardPair.getPlayedCard());
        }
        return cards;
    }

    @Transient
    private GamePlayerRel getChosenGamePlayerRel(Player p) {
        for (GamePlayerRel gamePlayerRel : gamePlayerRels) {
            if (gamePlayerRel.getPlayer().getId().longValue() == p.getId().longValue()) {
                return gamePlayerRel;
            }
        }
        return null;
    }

    public Round currentRound() {
        return rounds.get(rounds.size() - 1);
    }

    public int requiredPlayersCount() {
        return OHELL_PLAYERS_REQUIRED;
    }

    public void makeAnnounce(Player p, Announce a) {
        currentRound().updateRoundWithPlayerAnnounce(p, a);
    }

    public void initNextRound() {
        int nextRoundIndex = OhellRulesManager.getNextRoundIndex(currentRound().getRoundNumber());
        int nextDealerIndex = (currentRound().getIndexOfDealerPlayer() + 1) % OhellRulesManager.PLAYERS_COUNT;
        Round r = new Round(nextRoundIndex, dealer, gamePlayerRels, nextDealerIndex);
        rounds.add(r);
    }
}

