package control;

import model.interfaces.commandTypes.DirectionArgs;
import model.interfaces.commandTypes.MoveArgs;
import model.interfaces.commandTypes.NoArgs;
import model.interfaces.commandTypes.ArmyArgs;
import model.interfaces.commandTypes.ArgumentsVisitor;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import model.GameListener;

/**
 * Abstract class from which the other key managers inherit. Forces the
 * implementation of GameListener and ArgumentsVisitor. Ensures that the
 * activation state is never out of sync, and provides extensions for subclasses
 * for activation/deactivation behavior.
 * 
 * @author Ross Nichols
 */

abstract class GameKeyAdapter extends KeyAdapter implements GameListener, ArgumentsVisitor
{
    /**
     * Methods and attributes related to the activation state
     **/
    private boolean activated = false;
    
    final protected void activate()
    {
        if ( !activated )
        {
            activated = true;
            doActivation();
        }
    }
    
    abstract protected void doActivation();
    
    final protected void deactivate()
    {
        if ( activated )
        {
            activated = false;
            doDeactivation();
        }
    }
    
    abstract protected void doDeactivation();
    
    final protected boolean activated()
    {
        return activated;
    }
    
    /**
     * defer overriding of KeyAdapter to subclasses
     **/
    abstract public void keyPressed( KeyEvent e );
    
    /**
     * defer implementation of GameListener to subclasses
     **/
    abstract public void onTick();
    
    abstract public void onGameBegin();
    
    abstract public void onGameLose();
    
    abstract public void onGameWin();
    
    /**
     * defer implementation of ArgumentsVisitor to subclasses
     **/
    abstract public void visitMoveArgs( MoveArgs args );
    
    abstract public void visitArmyArgs( ArmyArgs args );
    
    abstract public void visitDirectionArgs( DirectionArgs args );
    
    abstract public void visitNoArgs( NoArgs args );
}
