package presentation.rest;

import com.sun.jersey.spi.resource.Singleton;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import presentation.twitter.Comment;
import presentation.twitter.Tweet;

import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.util.List;

@Singleton
@Path("tweets/binding")
public class TweetBindingResource {

	private final static Logger logger = LoggerFactory.getLogger(TweetBindingResource.class);

	@GET
	@Path("{username: [a-zA-Z][a-zA-Z_0-9]*}/stream")
	@Produces({MediaType.APPLICATION_JSON, ServiceLocator.MEDIA_TYPE_X_JAVASCRIPT})
	public Response stream(
			@Context HttpHeaders headers,
			@PathParam("username") String username,
			@QueryParam("callback") @DefaultValue("stream") String callback
	) {
		List<Tweet> entity = ServiceLocator.TWITTER.findStream(username);
		logger.info("Found stream username = {}.", username);
		return ResponseBuilder.newBuilder()
				.setHeaders(headers)
				.setCallback(callback)
				.setEntity(entity)
				.build();
	}

	@GET
	@Path("{username: [a-zA-Z][a-zA-Z_0-9]*}")
	@Produces({MediaType.APPLICATION_JSON, ServiceLocator.MEDIA_TYPE_X_JAVASCRIPT})
	public Response tweets(
			@Context HttpHeaders headers,
			@PathParam("username") String username,
			@QueryParam("callback") @DefaultValue("stream") String callback
	) {
		List<Tweet> entity = ServiceLocator.TWITTER.findTweets(username);
		logger.info("Found tweets username = {}.", username);
		return ResponseBuilder.newBuilder()
				.setHeaders(headers)
				.setCallback(callback)
				.setEntity(entity)
				.build();
	}

	@POST
	@Consumes(MediaType.APPLICATION_JSON)
	public Response create(Tweet.Builder tweet) {
		ServiceLocator.TWITTER.addTweet(tweet);
		logger.info("Created tweet = {}.", tweet);
		return Response.ok().build();
	}

	@PUT
	@Path("{tweetId}")
	@Consumes(MediaType.APPLICATION_JSON)
	public Response update(@PathParam("tweetId") long tweetId, Comment.Builder comment) {
		ServiceLocator.TWITTER.addComment(comment.withTweetId(tweetId).build());
		logger.info("Created tweet with comment = {}", comment);
		return Response.ok().build();
	}

	@DELETE
	@Path("{username: [a-zA-Z][a-zA-Z_0-9]*}")
	public Response remove(@PathParam("username") String username) {
		ServiceLocator.TWITTER.removeUsername(username);
		logger.info("Removed username = {}.", username);
		return Response.ok(Response.Status.OK).build();
	}

	@POST
	@Path("{username: [a-zA-Z][a-zA-Z_0-9]*}/follow")
	@Consumes(MediaType.APPLICATION_JSON)
	public Response follow(@PathParam("username") String username, List<String> usernames) {
		ServiceLocator.TWITTER.follow(username, usernames);
		logger.info("Username = {} will follow:  {}.", username, usernames);
		return Response.ok().build();
	}
}
