package presentation.twitter;

import org.codehaus.jackson.annotate.JsonIgnore;
import org.codehaus.jackson.annotate.JsonProperty;

import java.io.Serializable;
import java.util.Date;

public class Comment implements Comparable<Comment>, Serializable {

	@JsonIgnore
	private final long tweetId;

	@JsonProperty("username")
	private final String username;

	@JsonProperty("comment")
	private final String comment;

	@JsonProperty("date")
	private final Date date;

	public Comment(long tweetId, String username, String comment, Date date) {
		this.tweetId = tweetId;
		this.username = username;
		this.comment = comment;
		this.date = date == null ? new Date() : date;
	}

	public long getTweetId() {
		return tweetId;
	}

	public String getUsername() {
		return username;
	}

	public String getComment() {
		return comment;
	}

	public Date getDate() {
		return date;
	}

	@Override
	public int compareTo(Comment other) {

		int comparison = -date.compareTo(other.date);

		if (comparison == 0) {
			comparison = username.compareTo(other.username);

			if (comparison == 0) {
				comparison = comment.compareTo(other.comment);

				if (comparison == 0) {
					comparison = Long.valueOf(tweetId).compareTo(other.tweetId);
				}
			}
		}

		return comparison;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null || getClass() != obj.getClass()) {
			return false;
		}

		Comment tweet = (Comment) obj;

		if (tweetId != tweet.tweetId) {
			return false;
		}

		return true;
	}

	@Override
	public int hashCode() {
		return (int) (tweetId ^ (tweetId >>> 32));
	}

	@Override
	public String toString() {
		final StringBuilder sb = new StringBuilder();
		sb.append("Comment");
		sb.append("{tweetId=").append(tweetId);
		sb.append(", username='").append(username).append('\'');
		sb.append(", comment='").append(comment).append('\'');
		sb.append(", date=").append(date);
		sb.append('}');
		return sb.toString();
	}

	public static Builder newBuilder() {
		return new Builder();
	}

	public static class Builder {

		private long tweetId;

		@JsonProperty("username")
		private String username;

		@JsonProperty("comment")
		private String comment;

		@JsonProperty("date")
		private Date date;

		private Builder() {
		}

		public long getTweetId() {
			return tweetId;
		}

		public void setTweetId(long tweetId) {
			this.tweetId = tweetId;
		}

		public Builder withTweetId(long tweetId) {
			this.tweetId = tweetId;
			return this;
		}

		public String getUsername() {
			return username;
		}

		public void setUsername(String username) {
			this.username = username;
		}

		public Builder withUsername(String username) {
			this.username = username;
			return this;
		}

		public String getComment() {
			return comment;
		}

		public void setComment(String comment) {
			this.comment = comment;
		}

		public Builder withComment(String comment) {
			this.comment = comment;
			return this;
		}

		public Date getDate() {
			return date;
		}

		public void setDate(Date date) {
			this.date = date;
		}

		public Builder withDate(Date date) {
			this.date = date;
			return this;
		}

		public Comment build() {
			return new Comment(tweetId, username, comment, date);
		}

		@Override
		public String toString() {
			final StringBuilder sb = new StringBuilder();
			sb.append("Builder");
			sb.append("{tweetId=").append(tweetId);
			sb.append(", username='").append(username).append('\'');
			sb.append(", comment='").append(comment).append('\'');
			sb.append(", date=").append(date);
			sb.append('}');
			return sb.toString();
		}
	}
}
